<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\commerce_product\Entity\ProductInterface;

/**
 * Service to manage the image fix queue for products with failed image downloads.
 */
class ImageFixQueue
{
  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The remote media manager service.
   *
   * @var \Drupal\sogan_commerce_product\Service\RemoteMediaManager
   */
  protected $remoteMediaManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Maximum retry duration in seconds (24 hours).
   */
  const MAX_RETRY_DURATION = 86400;

  /**
   * Maximum number of retry attempts.
   */
  const MAX_ATTEMPTS = 72; // 24 hours with hourly cron

  /**
   * Constructs a new ImageFixQueue object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\sogan_commerce_product\Service\RemoteMediaManager $remote_media_manager
   *   The remote media manager service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(Connection $database, RemoteMediaManager $remote_media_manager, EntityTypeManagerInterface $entity_type_manager, LoggerChannelFactoryInterface $logger_factory)
  {
    $this->database = $database;
    $this->remoteMediaManager = $remote_media_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->logger = $logger_factory->get('sogan_commerce_product');
  }

  /**
   * Add a product to the image fix queue.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $product
   *   The product entity.
   * @param array $image_urls
   *   Array of failed image URLs.
   * @param string $context
   *   Context: 'creation' or 'manual'.
   */
  public function addProduct(ProductInterface $product, array $image_urls, string $context = 'creation')
  {
    if (empty($image_urls)) {
      return;
    }

    // Check if product already in queue
    $existing = $this->database->select('sogan_commerce_image_queue', 'q')
      ->fields('q', ['id'])
      ->condition('product_id', $product->id())
      ->condition('status', ['pending', 'processing'], 'IN')
      ->execute()
      ->fetchField();

    if ($existing) {
      // Update existing entry with new URLs
      $this->database->update('sogan_commerce_image_queue')
        ->fields([
          'image_urls' => json_encode($image_urls),
          'context' => $context,
        ])
        ->condition('id', $existing)
        ->execute();
    } else {
      // Insert new entry
      $this->database->insert('sogan_commerce_image_queue')
        ->fields([
          'product_id' => $product->id(),
          'image_urls' => json_encode($image_urls),
          'attempts' => 0,
          'created' => time(),
          'last_attempt' => 0,
          'status' => 'pending',
          'context' => $context,
        ])
        ->execute();
    }

    $this->logger->info('Added product @pid to image fix queue with @count URLs', [
      '@pid' => $product->id(),
      '@count' => count($image_urls),
    ]);
  }

  /**
   * Process pending items in the queue.
   *
   * Called by cron to retry failed image downloads.
   *
   * @param int $limit
   *   Maximum number of items to process.
   */
  public function processPendingItems(int $limit = 10)
  {
    $cutoff_time = time() - self::MAX_RETRY_DURATION;

    // Get pending items that haven't exceeded max attempts and are within 24h window
    $items = $this->database->select('sogan_commerce_image_queue', 'q')
      ->fields('q')
      ->condition('status', 'pending')
      ->condition('attempts', self::MAX_ATTEMPTS, '<')
      ->condition('created', $cutoff_time, '>')
      ->range(0, $limit)
      ->execute()
      ->fetchAll();

    foreach ($items as $item) {
      $this->processQueueItem($item);
    }

    // Clean up old failed items (older than 24h)
    $this->cleanupOldItems();
  }

  /**
   * Process a single queue item.
   *
   * @param object $item
   *   The queue item from database.
   */
  protected function processQueueItem($item)
  {
    // Mark as processing
    $this->database->update('sogan_commerce_image_queue')
      ->fields(['status' => 'processing'])
      ->condition('id', $item->id)
      ->execute();

    $product_storage = $this->entityTypeManager->getStorage('commerce_product');
    $product = $product_storage->load($item->product_id);

    if (!$product) {
      // Product deleted, remove from queue
      $this->removeFromQueue($item->id);
      return;
    }

    $image_urls = json_decode($item->image_urls, TRUE);
    if (empty($image_urls)) {
      $this->removeFromQueue($item->id);
      return;
    }

    $failed_urls = [];
    $success_count = 0;
    $media_items = [];

    // Get existing media IDs to avoid duplicates
    $existing_media_ids = [];
    if ($product->hasField('field_product_images') && !$product->get('field_product_images')->isEmpty()) {
      foreach ($product->get('field_product_images') as $item_ref) {
        if ($item_ref->target_id) {
          $existing_media_ids[] = $item_ref->target_id;
        }
      }
    }

    $context = [
      'product' => $product,
    ];

    foreach ($image_urls as $url) {
      $media_id = $this->remoteMediaManager->createMediaFromRemoteUrl($url, $product->getTitle(), $context);

      if ($media_id) {
        if (!in_array($media_id, $existing_media_ids)) {
          $media_items[] = ['target_id' => $media_id];
        }
        $success_count++;
      } else {
        $failed_urls[] = $url;
      }
    }

    // Attach successfully downloaded images to product
    if (!empty($media_items) && $product->hasField('field_product_images')) {
      $current_images = $product->get('field_product_images')->getValue();
      $all_images = array_merge($current_images, $media_items);
      $product->set('field_product_images', $all_images);
      $product->save();
    }

    // Update queue item
    $attempts = (int) $item->attempts + 1;

    if (empty($failed_urls)) {
      // All images succeeded
      $this->database->update('sogan_commerce_image_queue')
        ->fields([
          'status' => 'completed',
          'attempts' => $attempts,
          'last_attempt' => time(),
        ])
        ->condition('id', $item->id)
        ->execute();

      $this->logger->info('Successfully downloaded all images for product @pid', [
        '@pid' => $item->product_id,
      ]);
    } else {
      // Some images still failed
      $this->database->update('sogan_commerce_image_queue')
        ->fields([
          'status' => 'pending',
          'attempts' => $attempts,
          'last_attempt' => time(),
          'image_urls' => json_encode($failed_urls),
        ])
        ->condition('id', $item->id)
        ->execute();

      $this->logger->info('Partial success for product @pid: @success/@total images downloaded', [
        '@pid' => $item->product_id,
        '@success' => $success_count,
        '@total' => count($image_urls),
      ]);
    }
  }

  /**
   * Clean up old items that have exceeded the retry window.
   */
  protected function cleanupOldItems()
  {
    $cutoff_time = time() - self::MAX_RETRY_DURATION;

    // Mark items as failed if they're older than 24h
    $this->database->update('sogan_commerce_image_queue')
      ->fields(['status' => 'failed'])
      ->condition('created', $cutoff_time, '<')
      ->condition('status', ['pending', 'processing'], 'IN')
      ->execute();

    // Delete completed and failed items older than 7 days
    $delete_cutoff = time() - (7 * 86400);
    $this->database->delete('sogan_commerce_image_queue')
      ->condition('created', $delete_cutoff, '<')
      ->condition('status', ['completed', 'failed'], 'IN')
      ->execute();
  }

  /**
   * Remove an item from the queue.
   *
   * @param int $id
   *   The queue item ID.
   */
  protected function removeFromQueue(int $id)
  {
    $this->database->delete('sogan_commerce_image_queue')
      ->condition('id', $id)
      ->execute();
  }

  /**
   * Get queue statistics.
   *
   * @return array
   *   Array with counts by status.
   */
  public function getQueueStats()
  {
    $stats = [
      'pending' => 0,
      'processing' => 0,
      'completed' => 0,
      'failed' => 0,
      'total' => 0,
    ];

    $result = $this->database->select('sogan_commerce_image_queue', 'q')
      ->fields('q', ['status'])
      ->execute();

    foreach ($result as $row) {
      if (isset($stats[$row->status])) {
        $stats[$row->status]++;
      }
      $stats['total']++;
    }

    return $stats;
  }
}
