<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\node\NodeInterface;

/**
 * Service for resolving text and image content from supplier nodes.
 *
 * Handles:
 * - Text selection (title, description) using configurable strategies
 * - Image selection and image count comparison
 */
class ContentResolver
{

    /**
     * Merge policy configuration.
     *
     * @var array
     */
    protected array $mergePolicy;

    /**
     * Constructs a ContentResolver object.
     *
     * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
     *   The config factory.
     */
    public function __construct(ConfigFactoryInterface $config_factory)
    {
        $config = $config_factory->get('sogan_commerce_product.settings');
        $this->mergePolicy = $config->get('merge_policy') ?? [];
    }

    /**
     * Select text value from supplier nodes based on configured strategy.
     *
     * @param array $nodes
     *   Array of supplier product nodes.
     * @param string $field_name
     *   Field name to extract text from ('title' or 'description').
     * @param string $strategy_key
     *   Key in merge policy config (e.g., 'product_name_strategy').
     *
     * @return string|null
     *   Selected text value or NULL.
     */
    public function selectTextByStrategy(array $nodes, string $field_name, string $strategy_key): ?string
    {
        if (empty($nodes)) {
            return NULL;
        }

        $strategy = $this->mergePolicy[$strategy_key] ?? 'newest';
        $values = [];

        foreach ($nodes as $node) {
            $value = $this->extractTextFromNode($node, $field_name);

            if ($value) {
                $values[] = [
                    'value' => $value,
                    'created' => $node->getCreatedTime(),
                    'nid' => $node->id(),
                    'weight' => $this->getSupplierWeight($node),
                ];
            }
        }

        if (empty($values)) {
            return NULL;
        }

        return $this->applyTextStrategy($values, $strategy);
    }

    /**
     * Extract text value from a node based on field name.
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     * @param string $field_name
     *   The field name ('title', 'description', or a specific field).
     *
     * @return string|null
     *   The extracted text or NULL.
     */
    protected function extractTextFromNode(NodeInterface $node, string $field_name): ?string
    {
        if ($field_name === 'title') {
            // For titles: AI title -> original title -> label
            if ($node->hasField('field_ai_title') && !$node->get('field_ai_title')->isEmpty()) {
                return (string) $node->get('field_ai_title')->value;
            }
            if ($node->hasField('field_original_product_title') && !$node->get('field_original_product_title')->isEmpty()) {
                return (string) $node->get('field_original_product_title')->value;
            }
            return $node->label();
        }

        if ($field_name === 'description') {
            // For description: AI description -> supplier description
            if ($node->hasField('field_ai_description') && !$node->get('field_ai_description')->isEmpty()) {
                return (string) $node->get('field_ai_description')->value;
            }
            if ($node->hasField('field_supplier_product_descripti') && !$node->get('field_supplier_product_descripti')->isEmpty()) {
                return (string) $node->get('field_supplier_product_descripti')->value;
            }
            return NULL;
        }

        // Generic field access
        if ($node->hasField($field_name) && !$node->get($field_name)->isEmpty()) {
            return (string) $node->get($field_name)->value;
        }

        return NULL;
    }

    /**
     * Apply text selection strategy.
     *
     * @param array $values
     *   Array of ['value', 'created', 'nid', 'weight'].
     * @param string $strategy
     *   Strategy name: newest, oldest, longest, shortest, weight.
     *
     * @return string
     *   The selected value.
     */
    protected function applyTextStrategy(array $values, string $strategy): string
    {
        switch ($strategy) {
            case 'oldest':
                usort($values, fn($a, $b) => $a['created'] <=> $b['created']);
                return $values[0]['value'];

            case 'newest':
                usort($values, fn($a, $b) => $b['created'] <=> $a['created']);
                return $values[0]['value'];

            case 'longest':
                usort($values, fn($a, $b) => mb_strlen($b['value']) <=> mb_strlen($a['value']));
                return $values[0]['value'];

            case 'shortest':
                usort($values, fn($a, $b) => mb_strlen($a['value']) <=> mb_strlen($b['value']));
                return $values[0]['value'];

            case 'weight':
                // Weight: Lower weight is higher priority (standard Drupal behavior).
                // Tie-breaker: Newest created.
                usort($values, function ($a, $b) {
                    if ($a['weight'] !== $b['weight']) {
                        return $a['weight'] <=> $b['weight'];
                    }
                    return $b['created'] <=> $a['created'];
                });
                return $values[0]['value'];

            default:
                return $values[0]['value'];
        }
    }

    /**
     * Get the weight of the supplier associated with the node.
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     *
     * @return int
     *   The supplier term weight, or 0 if not found.
     */
    protected function getSupplierWeight(NodeInterface $node): int
    {
        if ($node->hasField('field_supplier') && !$node->get('field_supplier')->isEmpty()) {
            $supplier = $node->get('field_supplier')->entity;
            if ($supplier instanceof \Drupal\taxonomy\TermInterface) {
                if ($supplier->hasField('field_supplier_weight') && !$supplier->get('field_supplier_weight')->isEmpty()) {
                    return (int) $supplier->get('field_supplier_weight')->value;
                }
            }
        }
        return 0;
    }

    /**
     * Get the image strategy from config.
     *
     * @return string
     *   The image strategy ('combine' or 'most_images').
     */
    public function getImageStrategy(): string
    {
        return $this->mergePolicy['product_image_strategy'] ?? 'combine';
    }

    /**
     * Count images available on a supplier node.
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     *
     * @return int
     *   Total count of images (media + remote URLs).
     */
    public function countNodeImages(NodeInterface $node): int
    {
        $count = 0;

        // Count existing media items
        if ($node->hasField('field_images') && !$node->get('field_images')->isEmpty()) {
            $count += count($node->get('field_images')->getValue());
        }

        // Count remote URLs (potential images)
        if ($node->hasField('field_supplier_image_urls') && !$node->get('field_supplier_image_urls')->isEmpty()) {
            $count += count($node->get('field_supplier_image_urls')->getValue());
        }

        return $count;
    }

    /**
     * Select nodes for image processing based on strategy.
     *
     * @param array $nodes
     *   Array of supplier product nodes.
     *
     * @return array
     *   Filtered array of nodes to process for images.
     */
    public function selectNodesForImages(array $nodes): array
    {
        $strategy = $this->getImageStrategy();

        if ($strategy === 'combine') {
            return $nodes;
        }

        if ($strategy === 'most_images') {
            $best_node = NULL;
            $max_count = -1;

            foreach ($nodes as $node) {
                $count = $this->countNodeImages($node);
                if ($count > $max_count) {
                    $max_count = $count;
                    $best_node = $node;
                }
            }

            return $best_node ? [$best_node] : [];
        }

        return $nodes;
    }
}
