<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\Component\Transliteration\TransliterationInterface;
use Drupal\node\NodeInterface;
use Drupal\commerce_product\Entity\ProductVariationInterface;

/**
 * Service for attribute extraction, normalization, and matching.
 *
 * Handles:
 * - Extracting attributes from supplier product nodes
 * - Normalizing attribute labels and values for comparison
 * - Grouping suppliers by attribute signatures
 * - Matching variations to attributes
 */
class AttributeResolver
{

    /**
     * The transliteration service.
     *
     * @var \Drupal\Component\Transliteration\TransliterationInterface
     */
    protected TransliterationInterface $transliteration;

    /**
     * Constructs an AttributeResolver object.
     *
     * @param \Drupal\Component\Transliteration\TransliterationInterface $transliteration
     *   The transliteration service.
     */
    public function __construct(TransliterationInterface $transliteration)
    {
        $this->transliteration = $transliteration;
    }

    /**
     * Normalize an attribute label for comparison.
     *
     * @param string $label
     *   The attribute label.
     *
     * @return string
     *   Normalized label (lowercase, trimmed, transliterated).
     */
    public function normalizeAttributeLabel(string $label): string
    {
        $normalized = $this->transliteration->transliterate($label, 'tr', '_');
        return strtolower(trim($normalized));
    }

    /**
     * Normalize an attribute value for comparison.
     *
     * @param string $value
     *   The attribute value.
     *
     * @return string
     *   Normalized value (lowercase, trimmed, transliterated).
     */
    public function normalizeAttributeValue(string $value): string
    {
        $normalized = $this->transliteration->transliterate($value, 'tr', '_');
        return strtolower(trim($normalized));
    }

    /**
     * Extract attributes from a supplier product node.
     *
     * Priority: field_ai_attributes > field_product_attributes
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     *
     * @return array
     *   Array with 'original' and 'normalized' attribute maps.
     *   Format: [
     *     'original' => ['Color' => 'Red', 'Size' => 'Large'],
     *     'normalized' => ['color' => 'red', 'size' => 'large']
     *   ]
     */
    public function extractAttributes(NodeInterface $node): array
    {
        $original = [];
        $field_name = NULL;

        // Priority: AI attributes first
        if ($node->hasField('field_ai_attributes') && !$node->get('field_ai_attributes')->isEmpty()) {
            $field_name = 'field_ai_attributes';
        } elseif ($node->hasField('field_product_attributes') && !$node->get('field_product_attributes')->isEmpty()) {
            $field_name = 'field_product_attributes';
        }

        if ($field_name) {
            $values = $node->get($field_name)->getValue();
            foreach ($values as $item) {
                if (isset($item['key']) && isset($item['value']) && !empty($item['key']) && !empty($item['value'])) {
                    $original[$item['key']] = $item['value'];
                }
            }
        }

        // Create normalized version
        $normalized = [];
        foreach ($original as $label => $value) {
            $norm_label = $this->normalizeAttributeLabel($label);
            $norm_value = $this->normalizeAttributeValue($value);
            $normalized[$norm_label] = $norm_value;
        }

        return [
            'original' => $original,
            'normalized' => $normalized,
        ];
    }

    /**
     * Create a unique signature string from normalized attributes.
     *
     * @param array $normalized_attrs
     *   Normalized attribute map ['color' => 'red', 'size' => 'l'].
     *
     * @return string
     *   Sortable signature string (e.g., "color:red|size:l").
     */
    public function createAttributeSignature(array $normalized_attrs): string
    {
        if (empty($normalized_attrs)) {
            return '_no_attributes_';
        }

        // Sort by key for consistent signatures
        ksort($normalized_attrs);

        $parts = [];
        foreach ($normalized_attrs as $label => $value) {
            $parts[] = $label . ':' . $value;
        }

        return implode('|', $parts);
    }

    /**
     * Group supplier products by their variation attribute signatures.
     *
     * Only uses attributes that:
     * 1. Are present in ALL supplier products
     * 2. Have DIFFERENT values across products (variation attributes)
     *
     * Ignores:
     * - Common attributes (same value for all)
     * - Sparse attributes (not present in all products)
     *
     * @param array $nodes
     *   Array of supplier product nodes.
     *
     * @return array
     *   Map of signature => group data.
     *   Each group contains: nodes, original_attrs, normalized_attrs, variation_attrs
     */
    public function groupSuppliersByAttributes(array $nodes): array
    {
        // First analyze to find which attributes are variation attributes
        $analysis = $this->analyzeAttributes($nodes);
        $variation_keys = array_keys($analysis['variation_attributes']);

        // Normalize variation keys for comparison
        $normalized_variation_keys = [];
        foreach ($variation_keys as $key) {
            $normalized_variation_keys[] = $this->normalizeAttributeLabel($key);
        }

        $groups = [];

        foreach ($nodes as $node) {
            $attrs = $this->extractAttributes($node);

            // Filter to only variation attributes
            $variation_attrs = [];
            foreach ($attrs['normalized'] as $key => $value) {
                if (in_array($key, $normalized_variation_keys)) {
                    $variation_attrs[$key] = $value;
                }
            }

            // Create signature from only variation attributes
            $signature = $this->createAttributeSignature($variation_attrs);

            if (!isset($groups[$signature])) {
                // Get original (non-normalized) variation attrs for display
                $original_variation_attrs = [];
                foreach ($attrs['original'] as $key => $value) {
                    $norm_key = $this->normalizeAttributeLabel($key);
                    if (in_array($norm_key, $normalized_variation_keys)) {
                        $original_variation_attrs[$key] = $value;
                    }
                }

                $groups[$signature] = [
                    'nodes' => [],
                    'original_attrs' => $attrs['original'],
                    'normalized_attrs' => $attrs['normalized'],
                    'variation_attrs' => $original_variation_attrs,
                ];
            }

            $groups[$signature]['nodes'][] = $node;
        }

        return $groups;
    }

    /**
     * Get a single attribute value from a node by key.
     *
     * Prioritizes field_ai_attributes over field_product_attributes.
     * Uses case-insensitive key matching.
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     * @param string $target_key
     *   The attribute key to find.
     *
     * @return string|null
     *   The attribute value or NULL if not found.
     */
    public function getAttributeValueFromNode(NodeInterface $node, string $target_key): ?string
    {
        $normalized_target = $this->normalizeAttributeLabel($target_key);

        // Priority: AI attributes first
        $field_name = NULL;
        if ($node->hasField('field_ai_attributes') && !$node->get('field_ai_attributes')->isEmpty()) {
            $field_name = 'field_ai_attributes';
        } elseif ($node->hasField('field_product_attributes') && !$node->get('field_product_attributes')->isEmpty()) {
            $field_name = 'field_product_attributes';
        }

        if (!$field_name) {
            return NULL;
        }

        $values = $node->get($field_name)->getValue();
        foreach ($values as $item) {
            if (isset($item['key']) && isset($item['value'])) {
                $normalized_key = $this->normalizeAttributeLabel($item['key']);
                if ($normalized_key === $normalized_target) {
                    return $item['value'];
                }
            }
        }

        return NULL;
    }

    /**
     * Extract commerce attributes from a variation as normalized map.
     *
     * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
     *   The product variation.
     *
     * @return array
     *   Normalized attribute map ['color' => 'red', 'size' => 'l'].
     */
    public function extractVariationAttributes(ProductVariationInterface $variation): array
    {
        $normalized = [];

        $field_definitions = $variation->getFieldDefinitions();

        foreach ($field_definitions as $field_name => $field_definition) {
            // Look for attribute fields (they start with 'attribute_')
            if (strpos($field_name, 'attribute_') !== 0) {
                continue;
            }

            if ($variation->hasField($field_name) && !$variation->get($field_name)->isEmpty()) {
                $attribute_value = $variation->get($field_name)->entity;

                if ($attribute_value) {
                    // Extract attribute label from field name
                    $attribute_label = substr($field_name, strlen('attribute_'));

                    // Get the attribute value label
                    $value_label = $attribute_value->label();

                    // Normalize both
                    $norm_label = $this->normalizeAttributeLabel($attribute_label);
                    $norm_value = $this->normalizeAttributeValue($value_label);

                    $normalized[$norm_label] = $norm_value;
                }
            }
        }

        return $normalized;
    }

    /**
     * Check if a variation matches the given normalized attributes.
     *
     * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
     *   The variation to check.
     * @param array $normalized_attrs
     *   Normalized attribute map to compare against.
     *
     * @return bool
     *   TRUE if attributes match, FALSE otherwise.
     */
    public function matchVariationByAttributes(ProductVariationInterface $variation, array $normalized_attrs): bool
    {
        $variation_attrs = $this->extractVariationAttributes($variation);

        // Create signatures for comparison
        $var_signature = $this->createAttributeSignature($variation_attrs);
        $supplier_signature = $this->createAttributeSignature($normalized_attrs);

        return $var_signature === $supplier_signature;
    }

    /**
     * Analyze attributes across multiple nodes.
     *
     * Categorizes attributes as:
     * - common_attributes: Same value across all nodes (taxonomy only)
     * - variation_attributes: Different values (commerce attribute + taxonomy)
     *
     * Uses extractAttributes() which prioritizes field_ai_attributes over
     * field_product_attributes.
     *
     * @param array $nodes
     *   Array of supplier product nodes.
     *
     * @return array
     *   Array with 'common_attributes' and 'variation_attributes'.
     */
    public function analyzeAttributes(array $nodes): array
    {
        // Matrix with both original and normalized values
        $matrix = [];

        // Collect data into a matrix using extractAttributes which handles
        // priority: field_ai_attributes > field_product_attributes
        foreach ($nodes as $nid => $node) {
            $attrs = $this->extractAttributes($node);
            foreach ($attrs['original'] as $key => $value) {
                $norm_key = $this->normalizeAttributeLabel($key);
                $norm_value = $this->normalizeAttributeValue($value);
                $matrix[$norm_key][$nid] = [
                    'original_key' => $key,
                    'original_value' => trim($value),
                    'normalized_value' => $norm_value,
                ];
            }
        }

        $total_nodes = count($nodes);

        $common_attributes = [];
        $variation_attributes = [];

        foreach ($matrix as $norm_key => $values_by_node) {
            $count_nodes_with_attr = count($values_by_node);

            // Skip sparse data (not present in all nodes)
            if ($count_nodes_with_attr < $total_nodes) {
                continue;
            }

            // Use normalized values for uniqueness comparison to avoid
            // case differences like "Su Bazlı" vs "Su bazlı" being treated
            // as different variation values.
            $normalized_values = array_column($values_by_node, 'normalized_value');
            $unique_normalized = array_unique($normalized_values);
            $count_unique = count($unique_normalized);

            // Get an original key for output (use first found)
            $first_entry = reset($values_by_node);
            $original_key = $first_entry['original_key'];

            // Same normalized value across all nodes -> common
            if ($count_unique === 1) {
                // Use first original value for output
                $common_attributes[$original_key] = $first_entry['original_value'];
            }
            // Different normalized values -> variation attribute
            else {
                // Collect unique original values (deduplicated by normalized value)
                $seen_normalized = [];
                $unique_originals = [];
                foreach ($values_by_node as $entry) {
                    if (!isset($seen_normalized[$entry['normalized_value']])) {
                        $seen_normalized[$entry['normalized_value']] = TRUE;
                        $unique_originals[] = $entry['original_value'];
                    }
                }
                $variation_attributes[$original_key] = $unique_originals;
            }
        }

        return [
            'common_attributes' => $common_attributes,
            'variation_attributes' => $variation_attributes,
        ];
    }
}
