<?php

namespace Drupal\sogan_commerce_product\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Component\Transliteration\TransliterationInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\commerce_product\Entity\ProductAttribute;
use Drupal\commerce_product\Entity\ProductAttributeValue;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\taxonomy\Entity\Term;
use Drupal\Component\Utility\Html;

/**
 * Service to dynamically manage Product and Variation attributes.
 */
class AttributeManager
{

    protected $entityTypeManager;
    protected $fieldManager;
    protected $transliteration;
    protected $moduleHandler;
    protected $attributeFieldManager;

    public function __construct(
        EntityTypeManagerInterface $entity_type_manager,
        EntityFieldManagerInterface $field_manager,
        TransliterationInterface $transliteration,
        ModuleHandlerInterface $module_handler,
        $attribute_field_manager = NULL
    ) {
        $this->entityTypeManager = $entity_type_manager;
        $this->fieldManager = $field_manager;
        $this->transliteration = $transliteration;
        $this->moduleHandler = $module_handler;
        $this->attributeFieldManager = $attribute_field_manager;
    }

    /**
     * PUBLIC: Sets attributes for a Product Variation.
     */
    public function setVariationAttributes($variation, array $attributes)
    {
        if (is_numeric($variation)) {
            $variation = $this->entityTypeManager->getStorage('commerce_product_variation')->load($variation);
        }
        if (!$variation) return;

        $needs_save = FALSE;
        $reload_needed = FALSE;

        foreach ($attributes as $item) {
            $result = $this->processSingleVariationAttribute($variation, $item);
            if ($result === 'field_created') {
                $reload_needed = TRUE;
                $needs_save = TRUE;
            } elseif ($result === TRUE) {
                $needs_save = TRUE;
            }
        }

        if ($reload_needed) {
            // Clear field definitions cache to ensure new fields are picked up
            // This MUST be done before save() because save() triggers generateTitle(),
            // which tries to access the new fields.
            $this->fieldManager->clearCachedFieldDefinitions();

            // If variation is new (unsaved), we must re-create it to pick up new fields
            // because ContentEntityBase caches field definitions internally.
            if ($variation->isNew()) {
                $data = $variation->toArray();
                // Remove keys that shouldn't be copied
                unset($data['variation_id']);
                unset($data['uuid']);

                // Re-create entity
                $variation = $this->entityTypeManager->getStorage('commerce_product_variation')->create($data);
                $variation->save();
            }

            // Reset cache to ensure we get a fresh entity with new field definitions
            $this->entityTypeManager->getStorage('commerce_product_variation')->resetCache([$variation->id()]);

            // Reload entity
            $variation = $this->entityTypeManager->getStorage('commerce_product_variation')->load($variation->id());

            if (!$variation) {
                return NULL;
            }

            // Retry assignment for all attributes (idempotent)
            foreach ($attributes as $item) {
                if ($this->processSingleVariationAttribute($variation, $item)) {
                    $needs_save = TRUE;
                }
            }
        }
        if ($needs_save) {
            $variation->save();
        }

        return $variation;
    }

    /**
     * PUBLIC: Sets taxonomies (attributes) for a Product.
     *
     * @return \Drupal\commerce_product\Entity\ProductInterface|null
     *   The product entity (possibly reloaded if fields were created).
     */
    public function setProductTaxonomies($product, array $taxonomies)
    {
        if (is_numeric($product)) {
            $product = $this->entityTypeManager->getStorage('commerce_product')->load($product);
        }
        if (!$product) return NULL;

        $needs_save = FALSE;
        $needs_reload = FALSE;

        foreach ($taxonomies as $item) {
            $result = $this->processSingleProductTaxonomy($product, $item);
            if ($result === 'field_created') {
                $needs_reload = TRUE;
                $needs_save = TRUE;
            } elseif ($result === TRUE) {
                $needs_save = TRUE;
            }
        }

        // Reload product if new fields were created
        if ($needs_reload) {
            // Clear field definitions cache to ensure new fields are picked up
            $this->fieldManager->clearCachedFieldDefinitions();

            // If product is new (unsaved), we must re-create it to pick up new fields
            if ($product->isNew()) {
                $data = $product->toArray();
                // Remove keys that shouldn't be copied
                unset($data['product_id']);
                unset($data['uuid']);

                // Re-create entity
                $product = $this->entityTypeManager->getStorage('commerce_product')->create($data);
                $product->save();
            }

            // Reset cache to ensure we get a fresh entity with new field definitions
            $this->entityTypeManager->getStorage('commerce_product')->resetCache([$product->id()]);

            $product_id = $product->id();
            $product = $this->entityTypeManager->getStorage('commerce_product')->load($product_id);

            // Reprocess to set values on reloaded entity
            foreach ($taxonomies as $item) {
                if (empty($item['name']) || empty($item['value'])) continue;

                $machine_name = $this->generateMachineName($item['name']);
                $field_name = 'field_tax_' . $machine_name;
                $term = $this->ensureTerm($machine_name, $item['value']);
                $this->appendValueToField($product, $field_name, $term->id());
            }
        }

        // Don't save here - let the caller save
        // This ensures the caller has the reloaded entity reference

        return $product;
    }

  /* ========================================================================
   * VARIATION WORKERS
   * ======================================================================== */

    /**
     * Processes a single variation attribute.
     * @return bool|string TRUE if value assigned, 'field_created' if field was created, FALSE otherwise.
     */
    protected function processSingleVariationAttribute($variation, array $item)
    {
        if (empty($item['name']) || empty($item['value'])) return FALSE;

        $machine_name = $this->generateMachineName($item['name']);
        $field_created = FALSE;

        // 1. Create Attribute Bundle (If not exists)
        if ($this->ensureAttributeBundle($machine_name, $item['name'])) {
            $field_created = TRUE;
            // Explicitly ensure field is created for this variation type
            // This handles cases where Commerce hook might not fire or we need it immediately.
            $attribute = $this->entityTypeManager->getStorage('commerce_product_attribute')->load($machine_name);
            if ($attribute) {
                $this->attributeFieldManager->createField($attribute, $variation->bundle());
            }
        }

        // 2. Create and Configure Field
        $field_name = 'attribute_' . $machine_name;
        if ($this->ensureVariationField($variation->bundle(), $field_name)) {
            $field_created = TRUE;
        }

        // 3. Create Value
        $value_entity = $this->ensureAttributeValue($machine_name, $item['value']);

        // 4. Assign to Variation
        // If field was just created, the entity doesn't know about it, so assignment will fail.
        if ($field_created) {
            return 'field_created';
        }

        return $this->assignValueToField($variation, $field_name, $value_entity->id());
    }

    protected function ensureAttributeBundle($id, $label)
    {
        $storage = $this->entityTypeManager->getStorage('commerce_product_attribute');
        if (!$storage->load($id)) {
            $storage->create([
                'id' => $id,
                'label' => $label,
                'elementType' => 'commerce_product_rendered_attribute',
            ])->save();
            $this->fieldManager->clearCachedFieldDefinitions();
            return TRUE;
        }
        return FALSE;
    }

    protected function ensureVariationField($bundle, $field_name)
    {
        $changed = FALSE;
        // Commerce automatically adds the field when an attribute is created,
        // but we must ensure it is NOT required.
        $field_config = FieldConfig::loadByName('commerce_product_variation', $bundle, $field_name);
        if ($field_config && $field_config->isRequired()) {
            $field_config->setRequired(FALSE);
            $field_config->save();
            $changed = TRUE;
        }

        // Configure Form Display
        $form_display = $this->entityTypeManager->getStorage('entity_form_display')->load('commerce_product_variation.' . $bundle . '.default');
        if ($form_display) {
            $component = $form_display->getComponent($field_name);
            // If component missing or not using correct widget
            if (!$component || $component['type'] !== 'entity_reference_autocomplete_tags') {
                $form_display->setComponent($field_name, [
                    'type' => 'entity_reference_autocomplete_tags',
                    'weight' => 2,
                    'region' => 'content',
                    'settings' => [
                        'match_operator' => 'CONTAINS',
                        'match_limit' => 10,
                        'size' => 60,
                        'placeholder' => '',
                    ],
                ]);
                $changed = TRUE;
            }

            // Assign to group_attributes if field_group module exists
            if ($this->moduleHandler->moduleExists('field_group')) {
                $group_name = 'group_attributes';
                $third_party_settings = $form_display->getThirdPartySettings('field_group');

                // Ensure group exists (user said it does, but safety check)
                if (!isset($third_party_settings[$group_name])) {
                    // Fallback creation if missing
                    $group_config = [
                        'group_name' => $group_name,
                        'entity_type' => 'commerce_product_variation',
                        'bundle' => $bundle,
                        'mode' => 'default',
                        'label' => 'Attributes Information',
                        'children' => [],
                        'parent_name' => '',
                        'weight' => 12,
                        'format_type' => 'details',
                        'format_settings' => [
                            'open' => FALSE,
                            'required_fields' => TRUE,
                            'id' => '',
                            'classes' => '',
                            'description' => '',
                        ],
                        'region' => 'content',
                    ];
                    $form_display->setThirdPartySetting('field_group', $group_name, $group_config);
                    $third_party_settings[$group_name] = $group_config;
                }

                $group = $third_party_settings[$group_name];
                if (!in_array($field_name, $group['children'])) {
                    $group['children'][] = $field_name;
                    $form_display->setThirdPartySetting('field_group', $group_name, $group);
                    $changed = TRUE;
                }
            }

            if ($changed) {
                $form_display->save();
            }
        }

        return $changed;
    }

    protected function ensureAttributeValue($attribute_id, $value_label)
    {
        $storage = $this->entityTypeManager->getStorage('commerce_product_attribute_value');
        $entities = $storage->loadByProperties([
            'attribute' => $attribute_id,
            'name' => trim($value_label),
        ]);

        if ($entities) {
            return reset($entities);
        }

        $entity = $storage->create([
            'attribute' => $attribute_id,
            'name' => trim($value_label),
        ]);
        $entity->save();
        return $entity;
    }

  /* ========================================================================
   * PRODUCT (TAXONOMY) WORKERS
   * ======================================================================== */

    /**
     * Processes a single product taxonomy (attribute).
     *
     * @return string|bool
     *   Returns 'field_created' if a new field was created, TRUE if value was appended, FALSE otherwise.
     */
    protected function processSingleProductTaxonomy($product, array $item)
    {
        if (empty($item['name']) || empty($item['value'])) return FALSE;

        $machine_name = $this->generateMachineName($item['name']);
        $field_name = 'field_tax_' . $machine_name;

        // 1. Create Vocabulary
        $this->ensureVocabulary($machine_name, $item['name']);

        // 2. Create Field (Unlike Commerce Attributes, we must create this manually)
        $field_was_created = $this->ensureProductField($product->bundle(), $field_name, $item['name'], $machine_name);

        // 3. Create Term
        $term = $this->ensureTerm($machine_name, $item['value']);

        // If field was just created, return special signal
        if ($field_was_created) {
            return 'field_created';
        }

        // 4. Append to Product (Supports multi-value)
        return $this->appendValueToField($product, $field_name, $term->id());
    }

    protected function ensureVocabulary($vid, $name)
    {
        $storage = $this->entityTypeManager->getStorage('taxonomy_vocabulary');
        if (!$storage->load($vid)) {
            $storage->create([
                'vid' => $vid,
                'name' => $name,
            ])->save();
        }
    }

    protected function ensureTerm($vid, $name)
    {
        $storage = $this->entityTypeManager->getStorage('taxonomy_term');
        $terms = $storage->loadByProperties([
            'vid' => $vid,
            'name' => trim($name),
        ]);

        if ($terms) {
            return reset($terms);
        }

        $term = $storage->create([
            'vid' => $vid,
            'name' => trim($name),
        ]);
        $term->save();
        return $term;
    }

    protected function ensureProductField($bundle, $field_name, $label, $target_vocab)
    {
        $entity_type = 'commerce_product';
        $field_created = FALSE;

        // A. Field Storage (Cardinality -1 for unlimited values)
        $storage = FieldStorageConfig::loadByName($entity_type, $field_name);
        if (!$storage) {
            FieldStorageConfig::create([
                'field_name' => $field_name,
                'entity_type' => $entity_type,
                'type' => 'entity_reference',
                'cardinality' => -1, // UNLIMITED
                'settings' => ['target_type' => 'taxonomy_term'],
            ])->save();
            $field_created = TRUE;
        }

        // B. Field Instance
        if (!FieldConfig::loadByName($entity_type, $bundle, $field_name)) {
            FieldConfig::create([
                'field_name' => $field_name,
                'entity_type' => $entity_type,
                'bundle' => $bundle,
                'label' => $label,
                'settings' => [
                    'handler' => 'default',
                    'handler_settings' => ['target_bundles' => [$target_vocab => $target_vocab]],
                ],
            ])->save();

            // Add to Form Display
            $display = $this->entityTypeManager->getStorage('entity_form_display')
                ->load($entity_type . '.' . $bundle . '.default');
            if ($display) {
                $display->setComponent($field_name, [
                    'type' => 'entity_reference_autocomplete_tags',
                    'weight' => 100,
                ]);

                // Handle Field Group: group_product_attributes
                if ($this->moduleHandler->moduleExists('field_group')) {
                    $group_name = 'group_product_attributes';
                    $third_party_settings = $display->getThirdPartySettings('field_group');

                    // 1. Create group if not exists
                    if (!isset($third_party_settings[$group_name])) {
                        $group_config = [
                            'group_name' => $group_name,
                            'entity_type' => $entity_type,
                            'bundle' => $bundle,
                            'mode' => 'default',
                            'label' => 'Product Attributes',
                            'children' => [],
                            'parent_name' => '',
                            'weight' => 99,
                            'format_type' => 'details',
                            'format_settings' => [
                                'open' => FALSE,
                                'required_fields' => TRUE,
                                'id' => '',
                                'classes' => '',
                                'description' => '',
                            ],
                            'region' => 'content',
                        ];
                        $display->setThirdPartySetting('field_group', $group_name, $group_config);
                        $third_party_settings[$group_name] = $group_config;
                    }

                    // 2. Add field to group children
                    $group = $third_party_settings[$group_name];
                    if (!in_array($field_name, $group['children'])) {
                        $group['children'][] = $field_name;
                        $display->setThirdPartySetting('field_group', $group_name, $group);
                    }
                }

                $display->save();
            }

            $this->fieldManager->clearCachedFieldDefinitions();
            $field_created = TRUE;
        }

        return $field_created;
    }

  /* ========================================================================
   * UTILITIES
   * ======================================================================== */

    /**
     * Assigns a single value (Overwrite).
     */
    protected function assignValueToField($entity, $field_name, $target_id)
    {
        if ($entity->hasField($field_name)) {
            $current_val = $entity->get($field_name)->target_id;
            if ($current_val != $target_id) {
                $entity->set($field_name, $target_id);
                return TRUE;
            }
        }
        return FALSE;
    }

    /**
     * Appends a value (Multi-value support).
     */
    protected function appendValueToField($entity, $field_name, $target_id)
    {
        if ($entity->hasField($field_name)) {
            $values = $entity->get($field_name)->getValue();
            $exists = FALSE;
            foreach ($values as $val) {
                if ($val['target_id'] == $target_id) {
                    $exists = TRUE;
                    break;
                }
            }

            if (!$exists) {
                $entity->get($field_name)->appendItem(['target_id' => $target_id]);
                return TRUE;
            }
        }
        return FALSE;
    }

    /**
     * Generates a machine name from a string.
     */
    protected function generateMachineName($string)
    {
        $transliterated = $this->transliteration->transliterate($string, 'en', '_');
        $transliterated = strtolower($transliterated);
        $transliterated = preg_replace('@[^a-z0-9_]+@', '_', $transliterated);
        // Truncate to 22 chars to allow for 'attribute_' or 'field_tax_' prefix (10 chars)
        // Total field name limit is 32 chars.
        return substr($transliterated, 0, 22);
    }
}
