<?php

namespace Drupal\sogan_commerce_product\Plugin\views\field;

use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\sogan_commerce_product\Service\SupplierProductFieldResolver;

/**
 * A handler to provide supplier suggested price for commerce products.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("supplier_suggested_price")
 */
class SupplierSuggestedPrice extends FieldPluginBase
{

  /**
   * The supplier product field resolver service.
   *
   * @var \Drupal\sogan_commerce_product\Service\SupplierProductFieldResolver
   */
  protected $supplierFieldResolver;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
  {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->supplierFieldResolver = $container->get('sogan_commerce_product.supplier_product_field_resolver');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function query()
  {
    // Leave empty to avoid adding fields to the query.
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions()
  {
    $options = parent::defineOptions();
    $options['display_format'] = ['default' => 'list'];
    $options['show_supplier_name'] = ['default' => TRUE];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state)
  {
    parent::buildOptionsForm($form, $form_state);

    $form['display_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Display format'),
      '#options' => [
        'list' => $this->t('Unordered list'),
        'inline' => $this->t('Inline (comma-separated)'),
      ],
      '#default_value' => $this->options['display_format'],
    ];

    $form['show_supplier_name'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show supplier name'),
      '#default_value' => $this->options['show_supplier_name'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values)
  {
    $product = $this->getEntity($values);

    if (!$product || !$product->id()) {
      return ['#markup' => ''];
    }

    $product_id = $product->id();

    // Get suggested price data
    $price_data = $this->supplierFieldResolver->getProductSupplierField($product_id, 'field_suggested_price');

    // Get supplier names if needed
    $supplier_data = [];
    if ($this->options['show_supplier_name']) {
      $supplier_data = $this->supplierFieldResolver->getProductSupplierField($product_id, 'field_supplier');
    }

    // Collect unique prices
    $prices = [];

    foreach ($price_data as $variation_key => $variation_info) {
      foreach ($variation_info['supplier_fields'] as $index => $supplier_field) {
        $price_value = $supplier_field['field_value'];

        if (!empty($price_value) && is_array($price_value)) {
          $number = $price_value[0]['number'] ?? NULL;
          $currency = $price_value[0]['currency_code'] ?? 'TRY';

          if ($number !== NULL) {
            $formatted_price = number_format((float)$number, 2) . ' ' . $currency;

            // Get supplier name if option is enabled
            $supplier_name = '';
            if ($this->options['show_supplier_name'] && isset($supplier_data[$variation_key]['supplier_fields'][$index])) {
              $supplier_field_data = $supplier_data[$variation_key]['supplier_fields'][$index]['field_value'];
              if (!empty($supplier_field_data) && is_array($supplier_field_data)) {
                $supplier_tid = $supplier_field_data[0]['target_id'] ?? NULL;
                if ($supplier_tid) {
                  $term = $this->entityTypeManager->getStorage('taxonomy_term')->load($supplier_tid);
                  if ($term) {
                    $supplier_name = $term->label();
                  }
                }
              }
            }

            // Create display text
            if (!empty($supplier_name)) {
              $display_text = $supplier_name . ': ' . $formatted_price;
            } else {
              $display_text = $formatted_price;
            }

            // Avoid duplicates
            if (!in_array($display_text, $prices)) {
              $prices[] = $display_text;
            }
          }
        }
      }
    }

    // Build the output
    if (empty($prices)) {
      return ['#markup' => ''];
    }

    // Render based on display format
    if ($this->options['display_format'] === 'inline') {
      return [
        '#markup' => implode(', ', $prices),
      ];
    } else {
      return [
        '#theme' => 'item_list',
        '#items' => array_map(function ($item) {
          return ['#markup' => $item];
        }, $prices),
        '#list_type' => 'ul',
      ];
    }
  }
}
