<?php

namespace Drupal\sogan_commerce_product\Plugin\views\field;

use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\sogan_commerce_product\Service\ProductTableBuilder;
use Drupal\Core\Url;
use Drupal\Core\Render\Markup;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * A handler to provide a table of supplier product information.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("supplier_products_table")
 */
class SupplierProductsTable extends FieldPluginBase
{

  /**
   * The product table builder.
   *
   * @var \Drupal\sogan_commerce_product\Service\ProductTableBuilder
   */
  protected $productTableBuilder;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
  {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->productTableBuilder = $container->get('sogan_commerce_product.product_table_builder');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function query()
  {
    // Leave empty to avoid adding fields to the query.
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions()
  {
    $options = parent::defineOptions();

    // Column visibility options
    $options['show_product_name'] = ['default' => TRUE];
    $options['show_supplier'] = ['default' => TRUE];
    $options['show_sku'] = ['default' => TRUE];
    $options['show_cost_price'] = ['default' => TRUE];
    $options['show_list_price'] = ['default' => TRUE];
    $options['show_suggested_price'] = ['default' => TRUE];
    $options['show_stock'] = ['default' => TRUE];
    $options['show_ai_status'] = ['default' => TRUE];

    // Column label options
    $options['label_product_name'] = ['default' => 'Product Name'];
    $options['label_supplier'] = ['default' => 'Supplier'];
    $options['label_sku'] = ['default' => 'SKU'];
    $options['label_cost_price'] = ['default' => 'Cost Price'];
    $options['label_list_price'] = ['default' => 'List Price'];
    $options['label_suggested_price'] = ['default' => 'Suggested Price'];
    $options['label_stock'] = ['default' => 'Stock'];
    $options['label_ai_status'] = ['default' => 'AI Status'];

    // Display options
    $options['show_column_labels'] = ['default' => TRUE];
    $options['link_sku'] = ['default' => TRUE];
    $options['show_only_instock'] = ['default' => FALSE];
    $options['compact_table'] = ['default' => FALSE];
    $options['show_in_lightbox'] = ['default' => FALSE];
    $options['show_badge'] = ['default' => TRUE];
    $options['table_class'] = ['default' => 'views-table'];

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state)
  {
    parent::buildOptionsForm($form, $form_state);

    $form['columns'] = [
      '#type' => 'details',
      '#title' => $this->t('Column Visibility'),
      '#open' => TRUE,
    ];

    $form['show_product_name'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Product Name'),
      '#default_value' => $this->options['show_product_name'],
      '#group' => 'columns',
    ];

    $form['show_supplier'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Supplier'),
      '#default_value' => $this->options['show_supplier'],
      '#group' => 'columns',
    ];

    $form['show_sku'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show SKU'),
      '#default_value' => $this->options['show_sku'],
      '#group' => 'columns',
    ];

    $form['show_cost_price'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Cost Price'),
      '#default_value' => $this->options['show_cost_price'],
      '#group' => 'columns',
    ];

    $form['show_list_price'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show List Price'),
      '#default_value' => $this->options['show_list_price'],
      '#group' => 'columns',
    ];

    $form['show_suggested_price'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Suggested Price'),
      '#default_value' => $this->options['show_suggested_price'],
      '#group' => 'columns',
    ];

    $form['show_stock'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Stock'),
      '#default_value' => $this->options['show_stock'],
      '#group' => 'columns',
    ];

    $form['show_ai_status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show AI Status'),
      '#default_value' => $this->options['show_ai_status'],
      '#group' => 'columns',
    ];

    $form['labels'] = [
      '#type' => 'details',
      '#title' => $this->t('Column Labels'),
      '#open' => FALSE,
    ];

    $form['show_column_labels'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show column labels'),
      '#default_value' => $this->options['show_column_labels'],
      '#group' => 'labels',
    ];

    $form['label_product_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Product Name label'),
      '#default_value' => $this->options['label_product_name'],
      '#group' => 'labels',
      '#states' => [
        'visible' => [
          ':input[name="options[show_column_labels]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['label_supplier'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Supplier label'),
      '#default_value' => $this->options['label_supplier'],
      '#group' => 'labels',
      '#states' => [
        'visible' => [
          ':input[name="options[show_column_labels]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['label_sku'] = [
      '#type' => 'textfield',
      '#title' => $this->t('SKU label'),
      '#default_value' => $this->options['label_sku'],
      '#group' => 'labels',
      '#states' => [
        'visible' => [
          ':input[name="options[show_column_labels]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['label_cost_price'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Cost Price label'),
      '#default_value' => $this->options['label_cost_price'],
      '#group' => 'labels',
      '#states' => [
        'visible' => [
          ':input[name="options[show_column_labels]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['label_list_price'] = [
      '#type' => 'textfield',
      '#title' => $this->t('List Price label'),
      '#default_value' => $this->options['label_list_price'],
      '#group' => 'labels',
      '#states' => [
        'visible' => [
          ':input[name="options[show_column_labels]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['label_suggested_price'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Suggested Price label'),
      '#default_value' => $this->options['label_suggested_price'],
      '#group' => 'labels',
      '#states' => [
        'visible' => [
          ':input[name="options[show_column_labels]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['label_stock'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Stock label'),
      '#default_value' => $this->options['label_stock'],
      '#group' => 'labels',
      '#states' => [
        'visible' => [
          ':input[name="options[show_column_labels]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['label_ai_status'] = [
      '#type' => 'textfield',
      '#title' => $this->t('AI Status label'),
      '#default_value' => $this->options['label_ai_status'],
      '#group' => 'labels',
      '#states' => [
        'visible' => [
          ':input[name="options[show_column_labels]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['display'] = [
      '#type' => 'details',
      '#title' => $this->t('Display Options'),
      '#open' => FALSE,
    ];

    $form['link_sku'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Link SKU to supplier URL'),
      '#default_value' => $this->options['link_sku'],
      '#group' => 'display',
    ];

    $form['show_only_instock'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show only in-stock items'),
      '#description' => $this->t('Only display supplier products with stock greater than 0.'),
      '#default_value' => $this->options['show_only_instock'],
      '#group' => 'display',
    ];

    $form['compact_table'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Compact table'),
      '#description' => $this->t('Reduce table padding and spacing for a more compact display.'),
      '#default_value' => $this->options['compact_table'],
      '#group' => 'display',
    ];

    $form['show_in_lightbox'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show table in a lightbox'),
      '#description' => $this->t('Display a hamburger icon that opens the table in a modal dialog when clicked.'),
      '#default_value' => $this->options['show_in_lightbox'],
      '#group' => 'display',
    ];

    $form['show_badge'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show count badge'),
      '#description' => $this->t('Display a badge with the number of items over the icon.'),
      '#default_value' => $this->options['show_badge'],
      '#group' => 'display',
      '#states' => [
        'visible' => [
          ':input[name="options[show_in_lightbox]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['table_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Table CSS class'),
      '#default_value' => $this->options['table_class'],
      '#group' => 'display',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values)
  {
    $product = $this->getEntity($values);

    if (!$product || !$product->id()) {
      return ['#markup' => ''];
    }

    // If lightbox mode is enabled, return hamburger icon
    if ($this->options['show_in_lightbox']) {
      return $this->buildLightboxTrigger($product->id());
    }

    // Build table using the service
    return $this->productTableBuilder->buildSupplierTable($product, $this->options);
  }

  /**
   * Build a lightbox trigger (hamburger icon) for the table.
   *
   * @param int $product_id
   *   The product ID.
   *
   * @return array
   *   Render array for the lightbox trigger.
   */
  protected function buildLightboxTrigger($product_id)
  {
    // Pass current options to the controller via query parameters
    $query = $this->options;

    $url = Url::fromRoute('sogan_commerce_product.table_modal', [
      'commerce_product' => $product_id,
      'type' => 'supplier',
    ], [
      'query' => $query,
    ]);

    // Get count
    $product = $this->entityTypeManager->getStorage('commerce_product')->load($product_id);
    $count = $this->productTableBuilder->getSupplierProductCount($product);

    // Build the trigger button (Truck icon)
    $icon_svg = '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect x="1" y="3" width="15" height="13"></rect><polygon points="16 8 20 8 23 11 23 16 16 16 16 8"></polygon><circle cx="5.5" cy="18.5" r="2.5"></circle><circle cx="18.5" cy="18.5" r="2.5"></circle></svg>';

    // Add badge
    $badge = '';
    if ($this->options['show_badge']) {
      $badge = '<span class="badge-count">' . $count . '</span>';
    }

    $trigger = [
      '#type' => 'link',
      '#title' => Markup::create($icon_svg . $badge),
      '#url' => $url,
      '#attributes' => [
        'class' => ['supplier-table-lightbox-trigger', 'supplier-trigger', 'use-ajax'],
        'data-dialog-type' => 'modal',
        'data-dialog-options' => json_encode([
          'width' => 'auto',
          'maxWidth' => '90%',
          'dialogClass' => 'supplier-products-lightbox-dialog',
        ]),
        'title' => $this->t('View @count supplier products', ['@count' => $count]),
        'aria-label' => $this->t('View @count supplier products', ['@count' => $count]),
      ],
      '#attached' => [
        'library' => [
          'core/drupal.dialog.ajax',
          'sogan_commerce_product/supplier-table-lightbox',
        ],
      ],
    ];

    return [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['supplier-table-lightbox-wrapper'],
      ],
      'trigger' => $trigger,
    ];
  }
}
