<?php

namespace Drupal\sogan_commerce_product\Plugin\views\field;

use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\sogan_commerce_product\Service\SupplierProductFieldResolver;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * A handler to provide supplier product information for commerce products.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("supplier_product_info")
 */
class SupplierProductInfo extends FieldPluginBase
{

  /**
   * The supplier product field resolver service.
   *
   * @var \Drupal\sogan_commerce_product\Service\SupplierProductFieldResolver
   */
  protected $supplierFieldResolver;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
  {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->supplierFieldResolver = $container->get('sogan_commerce_product.supplier_product_field_resolver');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function query()
  {
    // Leave empty to avoid adding fields to the query.
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions()
  {
    $options = parent::defineOptions();
    $options['display_format'] = ['default' => 'list'];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state)
  {
    parent::buildOptionsForm($form, $form_state);

    $form['display_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Display format'),
      '#options' => [
        'list' => $this->t('Unordered list'),
        'inline' => $this->t('Inline (comma-separated)'),
      ],
      '#default_value' => $this->options['display_format'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values)
  {
    $product = $this->getEntity($values);

    if (!$product || !$product->id()) {
      return ['#markup' => ''];
    }

    $product_id = $product->id();

    // Get supplier data for all fields we need
    $supplier_data = $this->supplierFieldResolver->getProductSupplierField($product_id, 'field_supplier');
    $sku_data = $this->supplierFieldResolver->getProductSupplierField($product_id, 'field_supplier_sku');
    $url_data = $this->supplierFieldResolver->getProductSupplierField($product_id, 'field_supplier_url');
    $stock_data = $this->supplierFieldResolver->getProductSupplierField($product_id, 'field_supplier_stock');

    // Combine all data by variation and supplier
    $combined_data = [];

    foreach ($supplier_data as $variation_key => $variation_info) {
      foreach ($variation_info['supplier_fields'] as $index => $supplier_field) {
        $supplier_product_id = $supplier_field['supplier_product_id'];

        // Get supplier name from taxonomy term
        $supplier_name = '';
        if (!empty($supplier_field['field_value']) && is_array($supplier_field['field_value'])) {
          $supplier_tid = $supplier_field['field_value'][0]['target_id'] ?? NULL;
          if ($supplier_tid) {
            $term = $this->entityTypeManager->getStorage('taxonomy_term')->load($supplier_tid);
            if ($term) {
              $supplier_name = $term->label();
            }
          }
        }

        // Get SKU
        $sku = '';
        if (isset($sku_data[$variation_key]['supplier_fields'][$index]['field_value'])) {
          $sku = $sku_data[$variation_key]['supplier_fields'][$index]['field_value'];
        }

        // Get URL
        $url = '';
        if (isset($url_data[$variation_key]['supplier_fields'][$index]['field_value'])) {
          $url_value = $url_data[$variation_key]['supplier_fields'][$index]['field_value'];
          if (is_array($url_value) && isset($url_value[0]['uri'])) {
            $url = $url_value[0]['uri'];
          } elseif (is_string($url_value)) {
            $url = $url_value;
          }
        }

        // Get Stock
        $stock = '';
        if (isset($stock_data[$variation_key]['supplier_fields'][$index]['field_value'])) {
          $stock = $stock_data[$variation_key]['supplier_fields'][$index]['field_value'];
        }

        // Create unique key for supplier-sku combination
        $unique_key = $supplier_name . '|||' . $sku;

        // Only add if we have both supplier name and SKU, and it's not a duplicate
        if (!empty($supplier_name) && !empty($sku) && !isset($combined_data[$unique_key])) {
          $combined_data[$unique_key] = [
            'supplier_name' => $supplier_name,
            'sku' => $sku,
            'url' => $url,
            'stock' => $stock,
          ];
        }
      }
    }

    // Build the output
    if (empty($combined_data)) {
      return ['#markup' => ''];
    }

    $items = [];
    foreach ($combined_data as $data) {
      // Build SKU display (with or without link)
      if (!empty($data['url'])) {
        $sku_display = '<a href="' . htmlspecialchars($data['url']) . '" target="_blank">' . htmlspecialchars($data['sku']) . '</a>';
      } else {
        $sku_display = htmlspecialchars($data['sku']);
      }

      // Build the full item text
      $item_text = htmlspecialchars($data['supplier_name']) . ': ' . $sku_display;

      // Add stock if available (including 0)
      if ($data['stock'] !== '' && $data['stock'] !== NULL) {
        $item_text .= ' (' . htmlspecialchars($data['stock']) . ')';
      }

      $items[] = $item_text;
    }

    // Render based on display format
    if ($this->options['display_format'] === 'inline') {
      return [
        '#markup' => implode(', ', $items),
      ];
    } else {
      return [
        '#theme' => 'item_list',
        '#items' => array_map(function ($item) {
          return ['#markup' => $item];
        }, $items),
        '#list_type' => 'ul',
      ];
    }
  }
}
