<?php

namespace Drupal\sogan_commerce_product\Plugin\views\field;

use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\sogan_commerce_product\Service\ProductTableBuilder;
use Drupal\Core\Url;
use Drupal\Core\Render\Markup;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * A handler to provide a table of product variations.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("product_variations_table")
 */
class ProductVariationsTable extends FieldPluginBase
{

  /**
   * The product table builder.
   *
   * @var \Drupal\sogan_commerce_product\Service\ProductTableBuilder
   */
  protected $productTableBuilder;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
  {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->productTableBuilder = $container->get('sogan_commerce_product.product_table_builder');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function query()
  {
    // Leave empty to avoid adding fields to the query.
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions()
  {
    $options = parent::defineOptions();

    // Column visibility options
    $options['show_variation_title'] = ['default' => TRUE];
    $options['show_sku'] = ['default' => TRUE];
    $options['show_price'] = ['default' => TRUE];
    $options['show_list_price'] = ['default' => TRUE];
    $options['show_cost_price'] = ['default' => TRUE];
    $options['show_stock'] = ['default' => TRUE];
    $options['show_logistics_code'] = ['default' => TRUE];

    // Column label options
    $options['label_variation_title'] = ['default' => 'Variation'];
    $options['label_sku'] = ['default' => 'SKU'];
    $options['label_price'] = ['default' => 'Price'];
    $options['label_list_price'] = ['default' => 'List Price'];
    $options['label_cost_price'] = ['default' => 'Cost Price'];
    $options['label_stock'] = ['default' => 'Stock'];
    $options['label_logistics_code'] = ['default' => 'Logistics Code'];

    // Display options
    $options['show_column_labels'] = ['default' => TRUE];
    $options['compact_table'] = ['default' => FALSE];
    $options['show_in_lightbox'] = ['default' => FALSE];
    $options['show_badge'] = ['default' => TRUE];
    $options['table_class'] = ['default' => 'views-table'];

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state)
  {
    parent::buildOptionsForm($form, $form_state);

    $form['columns'] = [
      '#type' => 'details',
      '#title' => $this->t('Column Visibility'),
      '#open' => TRUE,
    ];

    $form['show_variation_title'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Variation Title'),
      '#default_value' => $this->options['show_variation_title'],
      '#group' => 'columns',
    ];

    $form['show_sku'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show SKU'),
      '#default_value' => $this->options['show_sku'],
      '#group' => 'columns',
    ];

    $form['show_price'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Price'),
      '#default_value' => $this->options['show_price'],
      '#group' => 'columns',
    ];

    $form['show_list_price'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show List Price'),
      '#default_value' => $this->options['show_list_price'],
      '#group' => 'columns',
    ];

    $form['show_cost_price'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Cost Price'),
      '#default_value' => $this->options['show_cost_price'],
      '#group' => 'columns',
    ];

    $form['show_stock'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Stock'),
      '#default_value' => $this->options['show_stock'],
      '#group' => 'columns',
    ];

    $form['show_logistics_code'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Logistics Code'),
      '#default_value' => $this->options['show_logistics_code'],
      '#group' => 'columns',
    ];

    $form['labels'] = [
      '#type' => 'details',
      '#title' => $this->t('Column Labels'),
      '#open' => FALSE,
    ];

    $form['show_column_labels'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show column labels'),
      '#default_value' => $this->options['show_column_labels'],
      '#group' => 'labels',
    ];

    $form['label_variation_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Variation Title label'),
      '#default_value' => $this->options['label_variation_title'],
      '#group' => 'labels',
      '#states' => [
        'visible' => [
          ':input[name="options[show_column_labels]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['label_sku'] = [
      '#type' => 'textfield',
      '#title' => $this->t('SKU label'),
      '#default_value' => $this->options['label_sku'],
      '#group' => 'labels',
      '#states' => [
        'visible' => [
          ':input[name="options[show_column_labels]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['label_price'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Price label'),
      '#default_value' => $this->options['label_price'],
      '#group' => 'labels',
      '#states' => [
        'visible' => [
          ':input[name="options[show_column_labels]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['label_list_price'] = [
      '#type' => 'textfield',
      '#title' => $this->t('List Price label'),
      '#default_value' => $this->options['label_list_price'],
      '#group' => 'labels',
      '#states' => [
        'visible' => [
          ':input[name="options[show_column_labels]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['label_cost_price'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Cost Price label'),
      '#default_value' => $this->options['label_cost_price'],
      '#group' => 'labels',
      '#states' => [
        'visible' => [
          ':input[name="options[show_column_labels]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['label_stock'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Stock label'),
      '#default_value' => $this->options['label_stock'],
      '#group' => 'labels',
      '#states' => [
        'visible' => [
          ':input[name="options[show_column_labels]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['label_logistics_code'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Logistics Code label'),
      '#default_value' => $this->options['label_logistics_code'],
      '#group' => 'labels',
      '#states' => [
        'visible' => [
          ':input[name="options[show_column_labels]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['display'] = [
      '#type' => 'details',
      '#title' => $this->t('Display Options'),
      '#open' => FALSE,
    ];

    $form['compact_table'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Compact table'),
      '#description' => $this->t('Reduce table padding and spacing for a more compact display.'),
      '#default_value' => $this->options['compact_table'],
      '#group' => 'display',
    ];

    $form['show_in_lightbox'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show table in a lightbox'),
      '#description' => $this->t('Display a hamburger icon that opens the table in a modal dialog when clicked.'),
      '#default_value' => $this->options['show_in_lightbox'],
      '#group' => 'display',
    ];

    $form['show_badge'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show count badge'),
      '#description' => $this->t('Display a badge with the number of items over the icon.'),
      '#default_value' => $this->options['show_badge'],
      '#group' => 'display',
      '#states' => [
        'visible' => [
          ':input[name="options[show_in_lightbox]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['table_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Table CSS class'),
      '#default_value' => $this->options['table_class'],
      '#group' => 'display',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values)
  {
    $product = $this->getEntity($values);

    if (!$product || !$product->id()) {
      return ['#markup' => ''];
    }

    // If lightbox mode is enabled, return hamburger icon
    if ($this->options['show_in_lightbox']) {
      return $this->buildLightboxTrigger($product->id());
    }

    // Build table using the service
    return $this->productTableBuilder->buildVariationTable($product, $this->options);
  }

  /**
   * Build a lightbox trigger (hamburger icon) for the table.
   *
   * @param int $product_id
   *   The product ID.
   *
   * @return array
   *   Render array for the lightbox trigger.
   */
  protected function buildLightboxTrigger($product_id)
  {
    // Pass current options to the controller via query parameters
    $query = $this->options;

    $url = Url::fromRoute('sogan_commerce_product.table_modal', [
      'commerce_product' => $product_id,
      'type' => 'variation',
    ], [
      'query' => $query,
    ]);

    // Get count
    $product = $this->entityTypeManager->getStorage('commerce_product')->load($product_id);
    $count = $this->productTableBuilder->getVariationCount($product);

    // Build the trigger button (List icon)
    $icon_svg = '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><line x1="8" y1="6" x2="21" y2="6"></line><line x1="8" y1="12" x2="21" y2="12"></line><line x1="8" y1="18" x2="21" y2="18"></line><line x1="3" y1="6" x2="3.01" y2="6"></line><line x1="3" y1="12" x2="3.01" y2="12"></line><line x1="3" y1="18" x2="3.01" y2="18"></line></svg>';

    // Add badge
    $badge = '';
    if ($this->options['show_badge']) {
      $badge = '<span class="badge-count">' . $count . '</span>';
    }

    $trigger = [
      '#type' => 'link',
      '#title' => Markup::create($icon_svg . $badge),
      '#url' => $url,
      '#attributes' => [
        'class' => ['supplier-table-lightbox-trigger', 'variation-trigger', 'use-ajax'],
        'data-dialog-type' => 'modal',
        'data-dialog-options' => json_encode([
          'width' => 'auto',
          'maxWidth' => '90%',
          'dialogClass' => 'product-variations-lightbox-dialog',
        ]),
        'title' => $this->t('View @count variations', ['@count' => $count]),
        'aria-label' => $this->t('View @count variations', ['@count' => $count]),
      ],
      '#attached' => [
        'library' => [
          'core/drupal.dialog.ajax',
          'sogan_commerce_product/supplier-table-lightbox', // Reuse existing CSS/JS
        ],
      ],
    ];

    return [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['product-variations-table-lightbox-wrapper'],
      ],
      'trigger' => $trigger,
    ];
  }
}
