<?php

namespace Drupal\sogan_commerce_product\Plugin\views\field;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\Markup;

/**
 * A handler to provide product stock levels by location.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("product_stock")
 */
class ProductStock extends FieldPluginBase
{

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The stock service manager.
   *
   * @var object
   */
  protected $stockServiceManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
  {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->cache = $container->get('cache.default');
    $instance->renderer = $container->get('renderer');
    $instance->stockServiceManager = $container->get('commerce_stock.service_manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function query()
  {
    // Leave empty to avoid adding fields to the query.
  }

  /**
   * {@inheritdoc}
   */
  public function clickSortable()
  {
    // This field is not sortable
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions()
  {
    $options = parent::defineOptions();
    $options['display_format'] = ['default' => 'list'];
    $options['show_zero_stock'] = ['default' => FALSE];
    $options['show_location_name'] = ['default' => TRUE];
    $options['show_total_stock'] = ['default' => FALSE];
    // Ensure these are never null
    $options['alter']['alter_text'] = ['default' => FALSE];
    $options['alter']['text'] = ['default' => ''];
    $options['alter']['make_link'] = ['default' => FALSE];
    $options['alter']['path'] = ['default' => ''];
    $options['alter']['absolute'] = ['default' => FALSE];
    $options['alter']['external'] = ['default' => FALSE];
    $options['alter']['replace_spaces'] = ['default' => FALSE];
    $options['alter']['path_case'] = ['default' => 'none'];
    $options['alter']['trim_whitespace'] = ['default' => FALSE];
    $options['alter']['alt'] = ['default' => ''];
    $options['alter']['rel'] = ['default' => ''];
    $options['alter']['link_class'] = ['default' => ''];
    $options['alter']['prefix'] = ['default' => ''];
    $options['alter']['suffix'] = ['default' => ''];
    $options['alter']['target'] = ['default' => ''];
    $options['alter']['nl2br'] = ['default' => FALSE];
    $options['alter']['max_length'] = ['default' => 0];
    $options['alter']['word_boundary'] = ['default' => TRUE];
    $options['alter']['ellipsis'] = ['default' => TRUE];
    $options['alter']['more_link'] = ['default' => FALSE];
    $options['alter']['more_link_text'] = ['default' => ''];
    $options['alter']['more_link_path'] = ['default' => ''];
    $options['alter']['strip_tags'] = ['default' => FALSE];
    $options['alter']['trim'] = ['default' => FALSE];
    $options['alter']['preserve_tags'] = ['default' => ''];
    $options['alter']['html'] = ['default' => FALSE];
    $options['element_type'] = ['default' => ''];
    $options['element_class'] = ['default' => ''];
    $options['element_label_type'] = ['default' => ''];
    $options['element_label_class'] = ['default' => ''];
    $options['element_label_colon'] = ['default' => FALSE];
    $options['element_wrapper_type'] = ['default' => ''];
    $options['element_wrapper_class'] = ['default' => ''];
    $options['element_default_classes'] = ['default' => TRUE];
    $options['empty'] = ['default' => ''];
    $options['hide_empty'] = ['default' => FALSE];
    $options['empty_zero'] = ['default' => FALSE];
    $options['hide_alter_empty'] = ['default' => TRUE];
    $options['click_sort_column'] = ['default' => ''];
    $options['type'] = ['default' => ''];
    $options['settings'] = ['default' => []];
    $options['group_column'] = ['default' => ''];
    $options['group_columns'] = ['default' => []];
    $options['group_rows'] = ['default' => TRUE];
    $options['delta_limit'] = ['default' => 0];
    $options['delta_offset'] = ['default' => 0];
    $options['delta_reversed'] = ['default' => FALSE];
    $options['delta_first_last'] = ['default' => FALSE];
    $options['multi_type'] = ['default' => 'separator'];
    $options['separator'] = ['default' => ', '];
    $options['field_api_classes'] = ['default' => FALSE];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state)
  {
    parent::buildOptionsForm($form, $form_state);

    $form['display_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Display format'),
      '#options' => [
        'list' => $this->t('List'),
        'inline' => $this->t('Inline (comma-separated)'),
        'table' => $this->t('Table'),
      ],
      '#default_value' => $this->options['display_format'],
    ];

    $form['show_zero_stock'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show locations with zero stock'),
      '#default_value' => $this->options['show_zero_stock'],
    ];

    $form['show_location_name'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show location name'),
      '#description' => $this->t('Display the location name along with the stock quantity.'),
      '#default_value' => $this->options['show_location_name'],
    ];

    $form['show_total_stock'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show total stock'),
      '#description' => $this->t('Display the total stock count across all locations.'),
      '#default_value' => $this->options['show_total_stock'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values)
  {
    /** @var \Drupal\commerce_product\Entity\ProductInterface $product */
    $product = $this->getEntity($values);

    if (!$product) {
      return '';
    }

    // Build cache key based on product ID and display options
    $cache_key = 'product_stock:' . $product->id() . ':' . md5(serialize($this->options));

    // Try to get from cache
    $cache = $this->cache->get($cache_key);
    if ($cache && !empty($cache->data)) {
      return Markup::create($cache->data);
    }

    $variations = $product->getVariations();
    if (empty($variations)) {
      $cached_output = '';
      $this->cache->set($cache_key, $cached_output, time() + 300, [
        'commerce_product:' . $product->id(),
        'commerce_stock_location',
      ]);
      return $cached_output;
    }

    // Get stock service manager
    $stock_service_manager = $this->stockServiceManager;

    // Get all stock locations
    $location_storage = $this->entityTypeManager->getStorage('commerce_stock_location');
    $all_locations = $location_storage->loadMultiple();

    // Aggregate stock by location across all variations
    $stock_by_location = [];

    foreach ($variations as $variation) {
      $stock_service = $stock_service_manager->getService($variation);
      $stock_checker = $stock_service->getStockChecker();

      // Check if stock checker supports batch location query
      if (method_exists($stock_checker, 'getLocationsStockLevels')) {
        $levels = $stock_checker->getLocationsStockLevels($variation, $all_locations);

        foreach ($levels as $location_id => $data) {
          if (!isset($stock_by_location[$location_id])) {
            $stock_by_location[$location_id] = [
              'quantity' => 0,
              'location' => $all_locations[$location_id] ?? NULL,
            ];
          }
          // The getLocationsStockLevels method returns an array with 'qty' and 'transactions_qty'
          // Sum them up for the total quantity.
          $stock_by_location[$location_id]['quantity'] += (float) $data['qty'] + (float) $data['transactions_qty'];
        }
      } else {
        // Fallback to per-location queries
        foreach ($all_locations as $location) {
          $location_id = $location->id();

          if (!isset($stock_by_location[$location_id])) {
            $stock_by_location[$location_id] = [
              'quantity' => 0,
              'location' => $location,
            ];
          }

          if (method_exists($stock_checker, 'getLocationStockLevel')) {
            $data = $stock_checker->getLocationStockLevel($location_id, $variation);
            // Handle different return formats
            if (is_array($data)) {
              $qty = ($data['qty'] ?? 0) + ($data['transactions_qty'] ?? 0);
            } else {
              $qty = (float) $data;
            }
            $stock_by_location[$location_id]['quantity'] += $qty;
          }
        }
      }
    }

    // Build output based on display format
    $output = '';
    $show_location_name = $this->options['show_location_name'] ?? TRUE;
    $show_zero_stock = $this->options['show_zero_stock'] ?? FALSE;
    $show_total_stock = $this->options['show_total_stock'] ?? FALSE;
    $display_format = $this->options['display_format'] ?? 'list';

    $items = [];
    $total_stock = 0;

    foreach ($stock_by_location as $location_id => $data) {
      $quantity = $data['quantity'];
      $location = $data['location'];

      // Always add to total, even if not shown (unless you only want to sum shown items?)
      // Usually total stock means total available, regardless of display filter.
      // But if we filter by stock > 0, we might still want the total.
      // Let's sum everything for now.
      $total_stock += $quantity;

      if (!$show_zero_stock && $quantity <= 0) {
        continue;
      }

      if (!$location) {
        continue;
      }

      $location_name = $location->getName();

      if ($display_format === 'table') {
        $items[] = [
          'location' => $location_name,
          'quantity' => number_format($quantity, 0),
        ];
      } else {
        if ($show_location_name) {
          $items[] = '<span style="white-space: nowrap;">' . $location_name . ':&nbsp;' . number_format($quantity, 0) . '</span>';
        } else {
          $items[] = number_format($quantity, 0);
        }
      }
    }

    if ($display_format === 'table') {
      if (!empty($items) || $show_total_stock) {
        $header = [];
        if ($show_location_name) {
          $header[] = $this->t('Location');
        }
        $header[] = $this->t('Stock');

        $rows = [];
        foreach ($items as $item) {
          $row = [];
          if ($show_location_name) {
            $row[] = $item['location'];
          }
          $row[] = $item['quantity'];
          $rows[] = $row;
        }

        if ($show_total_stock) {
          $row = [];
          if ($show_location_name) {
            $row[] = ['data' => $this->t('Total'), 'style' => 'font-weight:bold'];
          } else {
            $row[] = ['data' => $this->t('Total'), 'style' => 'font-weight:bold'];
          }
          $row[] = ['data' => number_format($total_stock, 0), 'style' => 'font-weight:bold'];
          $rows[] = $row;
        }

        $table = [
          '#type' => 'table',
          '#header' => $header,
          '#rows' => $rows,
          '#attributes' => ['class' => ['product-stock-table']],
        ];
        // Render the table to a string
        $output = $this->renderer->render($table);
      }
    } else {
      if (!empty($items)) {
        if ($display_format === 'list') {
          $output = implode('<br>', $items);
        } else {
          $output = implode(', ', $items);
        }
      }

      if ($show_total_stock) {
        $total_str = '<span style="white-space: nowrap;"><strong>' . $this->t('Total:') . '&nbsp;' . number_format($total_stock, 0) . '</strong></span>';
        if (!empty($output)) {
          $output .= '<br>' . $total_str;
        } else {
          $output = $total_str;
        }
      }
    }

    // Cache the result for 5 minutes with appropriate cache tags
    $this->cache->set($cache_key, $output, time() + 300, [
      'commerce_product:' . $product->id(),
      'commerce_stock_location',
    ]);

    return Markup::create($output);
  }
}
