<?php

namespace Drupal\sogan_commerce_product\Plugin\views\field;

use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A handler to provide product pricing from variations.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("product_price")
 */
class ProductPrice extends FieldPluginBase
{

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
  {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function query()
  {
    // Leave empty to avoid adding fields to the query.
  }

  /**
   * {@inheritdoc}
   */
  public function clickSortable()
  {
    // This field aggregates prices from multiple variations and cannot be sorted.
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions()
  {
    $options = parent::defineOptions();
    $options['show_currency'] = ['default' => TRUE];
    $options['decimal_places'] = ['default' => 2];
    $options['show_list_price'] = ['default' => TRUE];
    $options['colorize_list_price'] = ['default' => FALSE];
    $options['colorize_price'] = ['default' => FALSE];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state)
  {
    parent::buildOptionsForm($form, $form_state);

    $form['show_list_price'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show list price'),
      '#description' => $this->t('Display list price with strikethrough before the regular price.'),
      '#default_value' => $this->options['show_list_price'],
    ];

    $form['show_currency'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show currency code'),
      '#default_value' => $this->options['show_currency'],
    ];

    $form['decimal_places'] = [
      '#type' => 'number',
      '#title' => $this->t('Decimal places'),
      '#default_value' => $this->options['decimal_places'],
      '#min' => 0,
      '#max' => 4,
    ];

    $form['colorize_list_price'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Colorize list price'),
      '#description' => $this->t('Display list price in red color.'),
      '#default_value' => $this->options['colorize_list_price'],
    ];

    $form['colorize_price'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Colorize price'),
      '#description' => $this->t('Display regular price in green color.'),
      '#default_value' => $this->options['colorize_price'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values)
  {
    $product = $this->getEntity($values);

    if (!$product || !$product->id()) {
      return ['#markup' => ''];
    }

    // Get all variations
    $variations = $product->getVariations();

    if (empty($variations)) {
      return ['#markup' => ''];
    }

    $list_prices = [];
    $prices = [];
    $currency = 'TRY';

    // Collect all prices from variations
    foreach ($variations as $variation) {
      // Get list price
      if ($variation->hasField('list_price') && !$variation->get('list_price')->isEmpty()) {
        $list_price_value = $variation->get('list_price')->first()->getValue();
        if (isset($list_price_value['number'])) {
          $list_prices[] = (float) $list_price_value['number'];
          if (isset($list_price_value['currency_code'])) {
            $currency = $list_price_value['currency_code'];
          }
        }
      }

      // Get regular price
      if ($variation->hasField('price') && !$variation->get('price')->isEmpty()) {
        $price_value = $variation->get('price')->first()->getValue();
        if (isset($price_value['number'])) {
          $prices[] = (float) $price_value['number'];
          if (isset($price_value['currency_code'])) {
            $currency = $price_value['currency_code'];
          }
        }
      }
    }

    if (empty($prices)) {
      return ['#markup' => ''];
    }

    $decimal_places = (int) $this->options['decimal_places'];
    $output_parts = [];

    // Format list price (strikethrough)
    if ($this->options['show_list_price'] && !empty($list_prices)) {
      $list_prices = array_unique($list_prices);
      sort($list_prices);

      if (count($list_prices) === 1) {
        $list_price_text = number_format($list_prices[0], $decimal_places);
      } else {
        $min = number_format(min($list_prices), $decimal_places);
        $max = number_format(max($list_prices), $decimal_places);
        $list_price_text = $min . ' - ' . $max;
      }

      if ($this->options['show_currency']) {
        // Get currency symbol from Currency entity
        $currency_storage = $this->entityTypeManager->getStorage('commerce_currency');
        $currency_entity = $currency_storage->load($currency);
        if ($currency_entity) {
          $currency_symbol = $currency_entity->getSymbol();
          $list_price_text = $currency_symbol . $list_price_text;
        }
      }

      // Apply color if enabled
      $list_price_style = '';
      if ($this->options['colorize_list_price']) {
        $list_price_style = ' style="color: #dc3545;"';
      }

      $output_parts[] = '<del' . $list_price_style . '>' . $list_price_text . '</del>';
    }

    // Format regular price
    $prices = array_unique($prices);
    sort($prices);

    if (count($prices) === 1) {
      $price_text = number_format($prices[0], $decimal_places);
    } else {
      $min = number_format(min($prices), $decimal_places);
      $max = number_format(max($prices), $decimal_places);
      $price_text = $min . ' - ' . $max;
    }

    if ($this->options['show_currency']) {
      // Get currency symbol from Currency entity
      $currency_storage = $this->entityTypeManager->getStorage('commerce_currency');
      $currency_entity = $currency_storage->load($currency);
      if ($currency_entity) {
        $currency_symbol = $currency_entity->getSymbol();
        $price_text = $currency_symbol . $price_text;
      }
    }

    // Apply color if enabled
    $price_style = '';
    if ($this->options['colorize_price']) {
      $price_style = ' style="color: #28a745;"';
    }

    $output_parts[] = '<strong' . $price_style . '>' . $price_text . '</strong>';

    // Use Markup to preserve HTML and inline styles
    // Wrap in nowrap span to ensure single line
    return [
      '#markup' => \Drupal\Core\Render\Markup::create('<span style="white-space: nowrap;">' . implode('&nbsp; ', $output_parts) . '</span>'),
    ];
  }
}
