<?php

namespace Drupal\sogan_commerce_product\Plugin\views\field;

use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A handler to provide product cost from variations.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("product_cost")
 */
class ProductCost extends FieldPluginBase
{

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
  {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function query()
  {
    // Leave empty to avoid adding fields to the query.
  }

  /**
   * {@inheritdoc}
   */
  public function clickSortable()
  {
    // This field aggregates costs from multiple variations and cannot be sorted.
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions()
  {
    $options = parent::defineOptions();
    $options['show_currency'] = ['default' => TRUE];
    $options['decimal_places'] = ['default' => 2];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state)
  {
    parent::buildOptionsForm($form, $form_state);

    $form['show_currency'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show currency code'),
      '#default_value' => $this->options['show_currency'],
    ];

    $form['decimal_places'] = [
      '#type' => 'number',
      '#title' => $this->t('Decimal places'),
      '#default_value' => $this->options['decimal_places'],
      '#min' => 0,
      '#max' => 4,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values)
  {
    $product = $this->getEntity($values);

    if (!$product || !$product->id()) {
      return ['#markup' => ''];
    }

    // Get all variations
    $variations = $product->getVariations();

    if (empty($variations)) {
      return ['#markup' => ''];
    }

    $costs = [];
    $currency = 'TRY';

    // Collect all cost prices from variations
    foreach ($variations as $variation) {
      if ($variation->hasField('field_cost_price') && !$variation->get('field_cost_price')->isEmpty()) {
        $cost_value = $variation->get('field_cost_price')->first()->getValue();
        if (isset($cost_value['number'])) {
          $costs[] = (float) $cost_value['number'];
          if (isset($cost_value['currency_code'])) {
            $currency = $cost_value['currency_code'];
          }
        }
      }
    }

    if (empty($costs)) {
      return ['#markup' => ''];
    }

    // Remove duplicates and sort
    $costs = array_unique($costs);
    sort($costs);

    $decimal_places = (int) $this->options['decimal_places'];

    // Format output based on number of unique values
    if (count($costs) === 1) {
      // Single value
      $output = number_format($costs[0], $decimal_places);
    } else {
      // Multiple different values - show range
      $min = number_format(min($costs), $decimal_places);
      $max = number_format(max($costs), $decimal_places);
      $output = $min . ' - ' . $max;
    }

    // Add currency if enabled
    if ($this->options['show_currency']) {
      // Get currency symbol from Currency entity
      $currency_storage = $this->entityTypeManager->getStorage('commerce_currency');
      $currency_entity = $currency_storage->load($currency);
      if ($currency_entity) {
        $currency_symbol = $currency_entity->getSymbol();
        $output = $currency_symbol . $output;
      }
    }

    // Use Markup to preserve HTML and inline styles
    // Wrap in nowrap span to ensure single line
    return [
      '#markup' => \Drupal\Core\Render\Markup::create('<span style="white-space: nowrap;">' . $output . '</span>'),
    ];
  }
}
