<?php

namespace Drupal\sogan_commerce_product\Plugin\views\field;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Drupal\Component\Render\MarkupInterface;
use Drupal\Core\Render\Markup;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\sogan_commerce_product\Service\SupplierProductFieldResolver;

/**
 * Displays combined AI enhancement status for all supplier products of a commerce product.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("combined_ai_status")
 */
class CombinedAiStatus extends FieldPluginBase
{

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * The supplier product field resolver.
     *
     * @var \Drupal\sogan_commerce_product\Service\SupplierProductFieldResolver
     */
    protected $supplierFieldResolver;

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected $configFactory;

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
        $instance->entityTypeManager = $container->get('entity_type.manager');
        $instance->supplierFieldResolver = $container->get('sogan_commerce_product.supplier_product_field_resolver');
        $instance->configFactory = $container->get('config.factory');
        return $instance;
    }

    /**
     * {@inheritdoc}
     */
    public function query()
    {
        // This is a computed field, no database query needed.
    }

    /**
     * {@inheritdoc}
     */
    public function render(ResultRow $values)
    {
        $product = $this->getEntity($values);

        if (!$product || !$product->id()) {
            return ['#markup' => ''];
        }

        $product_id = $product->id();

        // Get all supplier products for this commerce product.
        $supplier_data = $this->supplierFieldResolver->getProductSupplierProducts($product_id);

        // Flatten to unique supplier product nodes.
        $supplier_products = [];
        $seen_ids = [];
        foreach ($supplier_data as $variation_products) {
            foreach ($variation_products as $node) {
                $nid = $node->id();
                if (!isset($seen_ids[$nid])) {
                    $seen_ids[$nid] = TRUE;
                    $supplier_products[] = $node;
                }
            }
        }

        $total = count($supplier_products);
        if ($total === 0) {
            return ['#markup' => '-'];
        }

        // Count how many have at least title AND description enhanced.
        $enhanced_count = 0;
        foreach ($supplier_products as $node) {
            if ($this->isEnhanced($node)) {
                $enhanced_count++;
            }
        }

        return $this->renderStatus($enhanced_count, $total);
    }

    /**
     * Check if a supplier product is considered "enhanced".
     *
     * A product is "enhanced" if at least title AND description fields have values.
     *
     * @param \Drupal\node\NodeInterface $node
     *   The supplier product node.
     *
     * @return bool
     *   TRUE if enhanced, FALSE otherwise.
     */
    protected function isEnhanced($node)
    {
        $config = $this->configFactory->get('supplier_products_ai_rewrite.settings');

        $title_field = $config->get('title_target_field') ?? 'field_ai_title';
        $desc_field = $config->get('description_target_field') ?? 'field_ai_description';

        $has_title = $node->hasField($title_field) && !$node->get($title_field)->isEmpty();
        $has_desc = $node->hasField($desc_field) && !$node->get($desc_field)->isEmpty();

        // Consider enhanced if both title and description are populated.
        return $has_title && $has_desc;
    }

    /**
     * Render the combined status.
     *
     * @param int $enhanced
     *   Number of enhanced supplier products.
     * @param int $total
     *   Total number of supplier products.
     *
     * @return array
     *   Render array.
     */
    protected function renderStatus($enhanced, $total)
    {
        // Determine CSS class based on ratio.
        if ($enhanced === $total) {
            // All enhanced - green.
            $status_class = '';
        } elseif ($enhanced > 0) {
            // Some enhanced - yellow/partial.
            $status_class = 'partial';
        } else {
            // None enhanced - gray.
            $status_class = 'none';
        }

        $label = $enhanced . '/' . $total;
        $title = $this->t('@enhanced of @total supplier products enhanced', [
            '@enhanced' => $enhanced,
            '@total' => $total,
        ]);

        // Robot/AI icon SVG (matching stroke style of other icons)
        $icon_svg = '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect x="3" y="11" width="18" height="10" rx="2"/><circle cx="12" cy="5" r="2"/><path d="M12 7v4"/><line x1="8" y1="16" x2="8" y2="16"/><line x1="16" y1="16" x2="16" y2="16"/></svg>';

        // Badge
        $badge = '<span class="badge-count">' . $label . '</span>';

        // Build classes matching supplier-table-lightbox-trigger pattern
        $classes = ['supplier-table-lightbox-trigger', 'ai-status-trigger'];
        if ($status_class) {
            $classes[] = $status_class;
        }

        return [
            '#type' => 'html_tag',
            '#tag' => 'span',
            '#value' => Markup::create($icon_svg . $badge),
            '#attributes' => [
                'class' => $classes,
                'title' => $title,
            ],
            '#attached' => [
                'library' => ['sogan_commerce_product/supplier_table_lightbox'],
            ],
        ];
    }
}
