<?php

namespace Drupal\sogan_commerce_product\Plugin\Action;

use Drupal\Core\Action\ActionBase;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\commerce_product\Entity\ProductInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\sogan_commerce_product\Service\ImageFixQueue;

/**
 * Redownload and attach supplier images to a product.
 *
 * @Action(
 *   id = "sogan_commerce_redownload_product_images",
 *   label = @Translation("Redownload & Attach Supplier Images"),
 *   type = "commerce_product"
 * )
 */
class RedownloadProductImages extends ActionBase implements ContainerFactoryPluginInterface
{
  /**
   * The image fix queue service.
   *
   * @var \Drupal\sogan_commerce_product\Service\ImageFixQueue
   */
  protected $imageFixQueue;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a RedownloadProductImages object.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ImageFixQueue $image_fix_queue, MessengerInterface $messenger)
  {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->imageFixQueue = $image_fix_queue;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
  {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('sogan_commerce_product.image_fix_queue'),
      $container->get('messenger')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function execute($entity = NULL)
  {
    if (!$entity instanceof ProductInterface) {
      return;
    }

    // Collect all image URLs from supplier products referenced by this product's variations
    $image_urls = [];
    $variations = $entity->getVariations();

    foreach ($variations as $variation) {
      if (!$variation->hasField('field_source_supplier_products') || $variation->get('field_source_supplier_products')->isEmpty()) {
        continue;
      }

      foreach ($variation->get('field_source_supplier_products')->referencedEntities() as $supplier_node) {
        if (!$supplier_node->hasField('field_supplier_image_urls') || $supplier_node->get('field_supplier_image_urls')->isEmpty()) {
          continue;
        }

        foreach ($supplier_node->get('field_supplier_image_urls')->getValue() as $item) {
          $uri = $item['uri'] ?? ($item['value'] ?? NULL);
          if (!empty($uri) && !in_array($uri, $image_urls)) {
            $image_urls[] = $uri;
          }
        }
      }
    }

    if (!empty($image_urls)) {
      // Add to queue for redownload
      $this->imageFixQueue->addProduct($entity, $image_urls, 'manual');

      $this->messenger->addStatus(t('Product "@title" has been queued for image redownload (@count URLs).', [
        '@title' => $entity->getTitle(),
        '@count' => count($image_urls),
      ]));
    } else {
      $this->messenger->addWarning(t('No supplier image URLs found for product "@title".', [
        '@title' => $entity->getTitle(),
      ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE)
  {
    /** @var \Drupal\commerce_product\Entity\ProductInterface $object */
    $access = $object->access('update', $account, TRUE);
    return $return_as_object ? $access : $access->isAllowed();
  }
}
