<?php

namespace Drupal\sogan_commerce_product\Plugin\Action;

use Drupal\Core\Action\ActionBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\sogan_commerce_product\Service\ProductBuilder;

/**
 * Recalculates merged fields (prices, text, stock) based on current strategies.
 *
 * @Action(
 *   id = "sogan_commerce_product_recalculate_merged_fields",
 *   label = @Translation("Recalculate merged fields"),
 *   type = "commerce_product"
 * )
 */
class RecalculateMergedFields extends ActionBase implements ContainerFactoryPluginInterface
{
    /**
     * The product builder service.
     *
     * @var \Drupal\sogan_commerce_product\Service\ProductBuilder
     */
    protected $productBuilder;

    /**
     * Constructs a new RecalculateMergedFields object.
     *
     * @param array $configuration
     *   A configuration array containing information about the plugin instance.
     * @param string $plugin_id
     *   The plugin_id for the plugin instance.
     * @param mixed $plugin_definition
     *   The plugin implementation definition.
     * @param \Drupal\sogan_commerce_product\Service\ProductBuilder $product_builder
     *   The product builder service.
     */
    public function __construct(array $configuration, $plugin_id, $plugin_definition, ProductBuilder $product_builder)
    {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->productBuilder = $product_builder;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('sogan_commerce_product.product_builder')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function execute($product = NULL)
    {
        if (!$product) {
            return;
        }

        // 1. Recalculate Product Level Fields (Title, Body)
        $this->productBuilder->recalculateProductFields($product);

        // 2. Recalculate Variation Level Fields (Prices, Stock)
        foreach ($product->getVariations() as $variation) {
            // Passing empty array forces re-evaluation of existing suppliers
            // This triggers the full price/stock recalculation logic in ProductBuilder
            $this->productBuilder->updateVariationWithSuppliers($variation, []);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE)
    {
        /** @var \Drupal\commerce_product\Entity\ProductInterface $object */
        return $object->access('update', $account, $return_as_object);
    }
}
