<?php

namespace Drupal\sogan_commerce_product\Plugin\Action;

use Drupal\Core\Action\ActionBase;
use Drupal\Core\Action\ActionManager;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Messenger\MessengerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Merges commerce products by rebuilding from supplier products.
 *
 * @Action(
 *   id = "commerce_product_rebuild_merge",
 *   label = @Translation("Merge Commerce Products (Rebuild)"),
 *   type = "commerce_product"
 * )
 */
class RebuildMergeCommerceProduct extends ActionBase implements ContainerFactoryPluginInterface
{

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * The database connection.
     *
     * @var \Drupal\Core\Database\Connection
     */
    protected $database;

    /**
     * The logger.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * The messenger.
     *
     * @var \Drupal\Core\Messenger\MessengerInterface
     */
    protected $messenger;

    /**
     * The action manager.
     *
     * @var \Drupal\Core\Action\ActionManager
     */
    protected $actionManager;

    /**
     * Constructs a new RebuildMergeCommerceProduct object.
     */
    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        EntityTypeManagerInterface $entity_type_manager,
        Connection $database,
        LoggerChannelFactoryInterface $logger_factory,
        MessengerInterface $messenger,
        ActionManager $action_manager
    ) {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->entityTypeManager = $entity_type_manager;
        $this->database = $database;
        $this->logger = $logger_factory->get('sogan_commerce_product');
        $this->messenger = $messenger;
        $this->actionManager = $action_manager;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('entity_type.manager'),
            $container->get('database'),
            $container->get('logger.factory'),
            $container->get('messenger'),
            $container->get('plugin.manager.action')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function execute($product = NULL)
    {
        if (!$product) {
            return;
        }

        $this->messenger->addWarning($this->t('This action requires multiple products to be selected. Please select at least 2 products.'));
    }

    /**
     * {@inheritdoc}
     */
    public function executeMultiple(array $entities)
    {
        if (count($entities) < 2) {
            $this->messenger->addError($this->t('Please select at least 2 products to merge.'));
            return;
        }

        // Check order history
        $products_with_orders = [];
        $order_counts = [];

        foreach ($entities as $product) {
            $order_count = $this->getOrderCount($product);
            if ($order_count > 0) {
                $products_with_orders[] = $product;
                $order_counts[$product->id()] = $order_count;
            }
        }

        if (count($products_with_orders) > 1) {
            $product_list = array_map(function ($p) use ($order_counts) {
                return $p->getTitle() . ' (' . $order_counts[$p->id()] . ' orders)';
            }, $products_with_orders);

            $this->messenger->addError($this->t(
                'Cannot merge: @count products have existing orders. This would break order history. Products with orders: @list',
                [
                    '@count' => count($products_with_orders),
                    '@list' => implode(', ', $product_list),
                ]
            ));
            return;
        }

        // Determine content source
        $content_source = !empty($products_with_orders) ? $products_with_orders[0] : $entities[0];

        $this->logger->info('Starting merge operation for @count products. Content source: @source', [
            '@count' => count($entities),
            '@source' => $content_source->getTitle(),
        ]);

        // Collect all supplier products from all variations
        $all_supplier_products = [];
        $variation_count = 0;

        foreach ($entities as $product) {
            foreach ($product->getVariations() as $variation) {
                $variation_count++;
                $supplier_products = $this->entityTypeManager
                    ->getStorage('node')
                    ->loadByProperties([
                        'type' => 'supplier_product',
                        'field_associated_product_variati' => $variation->id(),
                    ]);

                foreach ($supplier_products as $sp) {
                    $all_supplier_products[$sp->id()] = $sp;
                }
            }
        }

        if (empty($all_supplier_products)) {
            $this->messenger->addError($this->t('No supplier products found for selected products. Cannot merge.'));
            return;
        }

        $this->logger->info('Found @count supplier products from @vcount variations', [
            '@count' => count($all_supplier_products),
            '@vcount' => $variation_count,
        ]);

        // Preserve content from source product
        $preserved_content = $this->preserveContent($content_source);

        // Get store IDs before deletion
        $store_ids = [];
        foreach ($entities as $product) {
            foreach ($product->getStores() as $store) {
                $store_ids[$store->id()] = $store->id();
            }
        }

        // Clear associations from supplier products
        foreach ($all_supplier_products as $sp) {
            $sp->set('field_associated_product_variati', NULL);
            $sp->save();
        }

        $this->logger->info('Cleared associations from @count supplier products', [
            '@count' => count($all_supplier_products),
        ]);

        // Delete all selected products
        $deleted_product_titles = [];
        foreach ($entities as $product) {
            $deleted_product_titles[] = $product->getTitle();
            $this->logger->info('Deleting product @id (@title)', [
                '@id' => $product->id(),
                '@title' => $product->getTitle(),
            ]);
            $product->delete();
        }

        // Execute merge & create action on supplier products
        try {
            $merge_action_plugin = $this->actionManager->createInstance('sogan_commerce_merge_create_product');
            $merge_action_plugin->executeMultiple(array_values($all_supplier_products));

            $this->logger->info('Executed merge & create action on @count supplier products', [
                '@count' => count($all_supplier_products),
            ]);
        } catch (\Exception $e) {
            $this->logger->error('Failed to execute merge action: @error', ['@error' => $e->getMessage()]);
            $this->messenger->addError($this->t('Failed to rebuild products: @error', ['@error' => $e->getMessage()]));
            return;
        }

        // Find newly created products and apply preserved content
        // Invalidate entity caches to ensure changes are visible immediately
        // instead of using sleep() which is unreliable
        $this->entityTypeManager->getStorage('commerce_product')->resetCache();
        $new_products = $this->findNewlyCreatedProducts($store_ids);

        if (empty($new_products)) {
            $this->messenger->addWarning($this->t('Products were merged but could not auto-apply preserved content. Please verify the result.'));
        } else {
            foreach ($new_products as $new_product) {
                $this->applyPreservedContent($new_product, $preserved_content);
            }

            $this->logger->info('Applied preserved content to @count newly created product(s)', [
                '@count' => count($new_products),
            ]);
        }

        // Success message
        $this->messenger->addStatus($this->t(
            'Successfully merged @count products from @sp_count supplier products. Content preserved from: @source',
            [
                '@count' => count($entities),
                '@sp_count' => count($all_supplier_products),
                '@source' => $content_source->getTitle(),
            ]
        ));
    }

    /**
     * Gets the count of orders for a product.
     */
    protected function getOrderCount($product)
    {
        $variation_ids = array_map(function ($v) {
            return $v->id();
        }, $product->getVariations());

        if (empty($variation_ids)) {
            return 0;
        }

        return (int) $this->database->select('commerce_order_item', 'oi')
            ->fields('oi', ['order_item_id'])
            ->condition('purchased_entity', $variation_ids, 'IN')
            ->countQuery()
            ->execute()
            ->fetchField();
    }

    /**
     * Preserves content from source product.
     */
    protected function preserveContent($product)
    {
        $content = [
            'title' => $product->getTitle(),
        ];

        if ($product->hasField('body') && !$product->get('body')->isEmpty()) {
            $content['body'] = $product->get('body')->getValue();
        }

        if ($product->hasField('field_product_category') && !$product->get('field_product_category')->isEmpty()) {
            $content['categories'] = $product->get('field_product_category')->getValue();
        }

        if ($product->hasField('field_product_brand') && !$product->get('field_product_brand')->isEmpty()) {
            $content['brands'] = $product->get('field_product_brand')->getValue();
        }

        if ($product->hasField('field_product_tag') && !$product->get('field_product_tag')->isEmpty()) {
            $content['tags'] = $product->get('field_product_tag')->getValue();
        }

        if ($product->hasField('field_images') && !$product->get('field_images')->isEmpty()) {
            $content['images'] = $product->get('field_images')->getValue();
        }

        return $content;
    }

    /**
     * Finds newly created products in specified stores.
     */
    protected function findNewlyCreatedProducts(array $store_ids)
    {
        $product_ids = $this->entityTypeManager
            ->getStorage('commerce_product')
            ->getQuery()
            ->condition('stores', array_values($store_ids), 'IN')
            ->condition('created', time() - 30, '>')
            ->accessCheck(FALSE)
            ->execute();

        if (empty($product_ids)) {
            return [];
        }

        return $this->entityTypeManager
            ->getStorage('commerce_product')
            ->loadMultiple($product_ids);
    }

    /**
     * Applies preserved content to a product.
     */
    protected function applyPreservedContent($product, array $content)
    {
        $product->setTitle($content['title']);

        if (isset($content['body']) && $product->hasField('body')) {
            $product->set('body', $content['body']);
        }

        if (isset($content['categories']) && $product->hasField('field_product_category')) {
            $product->set('field_product_category', $content['categories']);
        }

        if (isset($content['brands']) && $product->hasField('field_product_brand')) {
            $product->set('field_product_brand', $content['brands']);
        }

        if (isset($content['tags']) && $product->hasField('field_product_tag')) {
            $product->set('field_product_tag', $content['tags']);
        }

        if (isset($content['images']) && $product->hasField('field_images')) {
            $existing_images = array_column($product->get('field_images')->getValue(), 'target_id');
            foreach ($content['images'] as $image) {
                if (!in_array($image['target_id'], $existing_images)) {
                    $product->get('field_images')->appendItem($image['target_id']);
                }
            }
        }

        $product->save();

        $this->logger->info('Applied preserved content to product @id (@title)', [
            '@id' => $product->id(),
            '@title' => $product->getTitle(),
        ]);
    }

    /**
     * {@inheritdoc}
     */
    public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE)
    {
        /** @var \Drupal\commerce_product\Entity\ProductInterface $object */
        return $object->access('update', $account, $return_as_object);
    }
}
