<?php

namespace Drupal\sogan_commerce_product\Plugin\Action;

use Drupal\Core\Action\ActionBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\node\NodeInterface;
use Drupal\sogan_commerce_product\Service\ProductBuilder;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Merge multiple supplier products into a Commerce product.
 *
 * @Action(
 *   id = "sogan_commerce_merge_create_product",
 *   label = @Translation("Merge &amp; Create Commerce Product"),
 *   type = "node"
 * )
 */
class MergeCreateCommerceProduct extends ActionBase implements ContainerFactoryPluginInterface
{

    /**
     * The product builder service.
     *
     * @var \Drupal\sogan_commerce_product\Service\ProductBuilder
     */
    protected $productBuilder;

    /**
     * The logger.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * The messenger service.
     *
     * @var \Drupal\Core\Messenger\MessengerInterface
     */
    protected $messenger;

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected $configFactory;

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * The plugin manager for queue workers.
     *
     * @var \Drupal\Component\Plugin\PluginManagerInterface|null
     */
    protected $queueWorkerManager;

    /**
     * Constructs a new MergeCreateCommerceProduct object.
     */
    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        ProductBuilder $product_builder,
        LoggerChannelFactoryInterface $logger_factory,
        MessengerInterface $messenger,
        ConfigFactoryInterface $config_factory,
        EntityTypeManagerInterface $entity_type_manager,
        $queue_worker_manager = NULL
    ) {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->productBuilder = $product_builder;
        $this->logger = $logger_factory->get('sogan_commerce_product');
        $this->messenger = $messenger;
        $this->configFactory = $config_factory;
        $this->entityTypeManager = $entity_type_manager;
        $this->queueWorkerManager = $queue_worker_manager;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('sogan_commerce_product.product_builder'),
            $container->get('logger.factory'),
            $container->get('messenger'),
            $container->get('config.factory'),
            $container->get('entity_type.manager'),
            $container->get('plugin.manager.queue_worker', ContainerInterface::NULL_ON_INVALID_REFERENCE)
        );
    }

    /**
     * {@inheritdoc}
     */
    public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE)
    {
        /** @var \Drupal\node\NodeInterface $object */
        $result = $object->access('update', $account, TRUE)
            ->andIf($object->status->access('edit', $account, TRUE));

        return $return_as_object ? $result : $result->isAllowed();
    }

    /**
     * {@inheritdoc}
     */
    public function execute($entity = NULL)
    {
        // Do nothing - all logic in executeMultiple()
    }

    /**
     * {@inheritdoc}
     */
    public function executeMultiple(array $entities)
    {
        $this->logger->notice('MergeCreateCommerceProduct: Starting with @count entities', ['@count' => count($entities)]);

        // Check if AI enhancement should run before merging.
        $aiConfig = $this->configFactory->get('supplier_products_ai_rewrite.settings');
        $aiEnabled = (bool) $aiConfig->get('enabled');
        $enhanceOnMerge = (bool) $aiConfig->get('enhance_on_merge');

        if ($aiEnabled && $enhanceOnMerge && $this->queueWorkerManager) {
            $entities = $this->runAiEnhancementWithMerge($entities);
        }

        // Use the existing working method that creates all fields correctly.
        $product = $this->productBuilder->createProductFromSuppliers($entities);

        if ($product) {
            $this->messenger->addStatus($this->t('Successfully created product "@title" with @count variations from @suppliers supplier products.', [
                '@title' => $product->label(),
                '@count' => count($product->getVariations()),
                '@suppliers' => count($entities),
            ]));
        } else {
            $this->messenger->addError($this->t('Failed to create product from selected supplier products.'));
        }
    }

    /**
     * Runs AI enhancement on all entities with merge instructions appended.
     *
     * @param array $entities
     *   The supplier product nodes.
     *
     * @return array
     *   The reloaded entities after AI enhancement.
     */
    protected function runAiEnhancementWithMerge(array $entities): array
    {
        $aiConfig = $this->configFactory->getEditable('supplier_products_ai_rewrite.settings');

        // Store original context instructions.
        $originalContext = $aiConfig->get('context_instructions') ?? '';
        $mergeInstructions = $aiConfig->get('merge_instructions') ?? '';

        try {
            // Temporarily append merge instructions to context.
            if (!empty($mergeInstructions)) {
                $combinedContext = $originalContext . "\n\n" . $mergeInstructions;
                $aiConfig->set('context_instructions', $combinedContext)->save();
            }

            /** @var \Drupal\supplier_products_ai_rewrite\Plugin\QueueWorker\AIContentEnhancer $queueWorker */
            $queueWorker = $this->queueWorkerManager->createInstance('ai_content_enhancer');

            // Process all entities as a batch.
            foreach ($entities as $entity) {
                if (!$entity instanceof NodeInterface) {
                    continue;
                }

                $item = [
                    'nid' => $entity->id(),
                    'tasks' => ['title', 'description', 'attributes', 'categorize', 'brand'],
                ];

                $queueWorker->processItem($item);
            }

            // Flush batch to ensure all items are processed together.
            if (class_exists('\Drupal\supplier_products_ai_rewrite\Plugin\QueueWorker\AIContentEnhancer')) {
                \Drupal\supplier_products_ai_rewrite\Plugin\QueueWorker\AIContentEnhancer::flushBatch();
            }

            // Reload all entities to get AI-enhanced fields.
            $reloadedEntities = [];
            $nodeStorage = $this->entityTypeManager->getStorage('node');
            foreach ($entities as $entity) {
                if ($entity instanceof NodeInterface) {
                    $reloadedEntities[] = $nodeStorage->load($entity->id());
                }
            }

            $this->logger->notice('AI enhancement completed for @count entities in merge mode', [
                '@count' => count($reloadedEntities),
            ]);

            return $reloadedEntities;
        } catch (\Exception $e) {
            $this->logger->warning('AI enhancement failed during merge: @message', [
                '@message' => $e->getMessage(),
            ]);
            // Return original entities if AI enhancement fails.
            return $entities;
        } finally {
            // Restore original context instructions.
            if (!empty($mergeInstructions)) {
                $aiConfig->set('context_instructions', $originalContext)->save();
            }
        }
    }
}
