<?php

namespace Drupal\sogan_commerce_product\Plugin\Action;

use Drupal\Core\Action\ActionBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\node\NodeInterface;
use Drupal\sogan_commerce_product\Service\ProductCreator;
use Drupal\Core\Access\AccessResult;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Create a Commerce Product from a supplier_product node.
 *
 * @Action(
 *   id = "sogan_commerce_create_product_from_supplier_node",
 *   label = @Translation("Create Commerce Product"),
 *   type = "node"
 * )
 */
class CreateCommerceProduct extends ActionBase implements ContainerFactoryPluginInterface
{
    /**
     * The product creator service.
     *
     * @var \Drupal\sogan_commerce_product\Service\ProductCreator
     */
    protected ProductCreator $productCreator;

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected ConfigFactoryInterface $configFactory;

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected EntityTypeManagerInterface $entityTypeManager;

    /**
     * The plugin manager for queue workers.
     *
     * @var \Drupal\Component\Plugin\PluginManagerInterface|null
     */
    protected $queueWorkerManager;

    /**
     * Constructs a new CreateCommerceProduct object.
     */
    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        ProductCreator $product_creator,
        ConfigFactoryInterface $config_factory,
        EntityTypeManagerInterface $entity_type_manager,
        $queue_worker_manager = NULL
    ) {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->productCreator = $product_creator;
        $this->configFactory = $config_factory;
        $this->entityTypeManager = $entity_type_manager;
        $this->queueWorkerManager = $queue_worker_manager;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('sogan_commerce_product.product_creator'),
            $container->get('config.factory'),
            $container->get('entity_type.manager'),
            $container->get('plugin.manager.queue_worker', ContainerInterface::NULL_ON_INVALID_REFERENCE)
        );
    }

    /**
     * {@inheritdoc}
     */
    public function execute($entity = NULL)
    {
        if (!$entity instanceof NodeInterface || $entity->bundle() !== 'supplier_product') {
            return;
        }

        // Check if AI enhancement should run before creating commerce product.
        $aiConfig = $this->configFactory->get('supplier_products_ai_rewrite.settings');
        $aiEnabled = (bool) $aiConfig->get('enabled');
        $enhanceOnCreate = (bool) $aiConfig->get('enhance_on_create');

        if ($aiEnabled && $enhanceOnCreate && $this->queueWorkerManager) {
            $this->runAiEnhancement($entity);
            // Reload entity to get AI-enhanced fields.
            $entity = $this->entityTypeManager->getStorage('node')->load($entity->id());
        }

        // Create product - messages are handled by ProductBuilder/ProductCreator services.
        $this->productCreator->createProductFromSupplier($entity);
    }

    /**
     * Runs AI enhancement on the supplier product.
     *
     * @param \Drupal\node\NodeInterface $entity
     *   The supplier product node.
     */
    protected function runAiEnhancement(NodeInterface $entity): void
    {
        try {
            /** @var \Drupal\supplier_products_ai_rewrite\Plugin\QueueWorker\AIContentEnhancer $queueWorker */
            $queueWorker = $this->queueWorkerManager->createInstance('ai_content_enhancer');

            $item = [
                'nid' => $entity->id(),
                'tasks' => ['title', 'description', 'attributes', 'categorize', 'brand'],
            ];

            $queueWorker->processItem($item);

            // Flush batch to ensure processing completes.
            if (class_exists('\Drupal\supplier_products_ai_rewrite\Plugin\QueueWorker\AIContentEnhancer')) {
                \Drupal\supplier_products_ai_rewrite\Plugin\QueueWorker\AIContentEnhancer::flushBatch();
            }
        } catch (\Exception $e) {
            // Log error but don't block product creation.
            \Drupal::logger('sogan_commerce_product')->warning('AI enhancement failed for node @nid: @message', [
                '@nid' => $entity->id(),
                '@message' => $e->getMessage(),
            ]);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE)
    {
        // Only allow action for supplier_product bundle nodes.
        $allowed = (
            $object instanceof NodeInterface && $object->bundle() === 'supplier_product'
        );

        return $return_as_object ? AccessResult::allowedIf($allowed) : $allowed;
    }
}
