<?php

namespace Drupal\sogan_commerce_product\Plugin\Action;

use Drupal\commerce_product\Entity\ProductInterface;
use Drupal\Core\Action\ActionBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\node\NodeInterface;
use Drupal\sogan_commerce_product\Service\ProductBuilder;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * AI Enhance source supplier products for commerce products.
 *
 * Differentiates between:
 * - Merged products (multiple sources): processes all sources together
 * - Simple products (single source): regular batch processing
 *
 * @Action(
 *   id = "ai_enhance_commerce_product",
 *   label = @Translation("AI Enhance Product"),
 *   type = "commerce_product"
 * )
 */
class AIEnhanceCommerceProduct extends ActionBase implements ContainerFactoryPluginInterface
{

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected EntityTypeManagerInterface $entityTypeManager;

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected ConfigFactoryInterface $configFactory;

    /**
     * The queue worker manager.
     *
     * @var \Drupal\Core\Queue\QueueWorkerManagerInterface|null
     */
    protected ?QueueWorkerManagerInterface $queueWorkerManager;

    /**
     * The logger factory.
     *
     * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
     */
    protected LoggerChannelFactoryInterface $loggerFactory;

    /**
     * The messenger service.
     *
     * @var \Drupal\Core\Messenger\MessengerInterface
     */
    protected $messenger;

    /**
     * The product builder service.
     *
     * @var \Drupal\sogan_commerce_product\Service\ProductBuilder
     */
    protected ProductBuilder $productBuilder;

    /**
     * Constructs a new AIEnhanceCommerceProduct object.
     */
    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        EntityTypeManagerInterface $entity_type_manager,
        ConfigFactoryInterface $config_factory,
        ?QueueWorkerManagerInterface $queue_worker_manager,
        LoggerChannelFactoryInterface $logger_factory,
        MessengerInterface $messenger,
        ProductBuilder $product_builder
    ) {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->entityTypeManager = $entity_type_manager;
        $this->configFactory = $config_factory;
        $this->queueWorkerManager = $queue_worker_manager;
        $this->loggerFactory = $logger_factory;
        $this->messenger = $messenger;
        $this->productBuilder = $product_builder;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('entity_type.manager'),
            $container->get('config.factory'),
            $container->has('plugin.manager.queue_worker') ? $container->get('plugin.manager.queue_worker') : NULL,
            $container->get('logger.factory'),
            $container->get('messenger'),
            $container->get('sogan_commerce_product.product_builder')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function execute($entity = NULL)
    {
        $this->executeMultiple([$entity]);
    }

    /**
     * {@inheritdoc}
     */
    public function executeMultiple(array $entities)
    {
        $logger = $this->loggerFactory->get('sogan_commerce_product');
        $aiConfig = $this->configFactory->get('supplier_products_ai_rewrite.settings');

        // Check if AI enhancement is enabled.
        if (!$aiConfig->get('enabled')) {
            $this->messenger->addWarning($this->t('AI enhancement is disabled in settings.'));
            return;
        }

        if (!$this->queueWorkerManager) {
            $this->messenger->addError($this->t('Queue worker manager not available.'));
            return;
        }

        // Categorize products.
        $mergedProducts = [];
        $simpleProducts = [];

        foreach ($entities as $product) {
            if (!$product instanceof ProductInterface) {
                continue;
            }

            $category = $this->categorizeProduct($product);
            if ($category === 'merged') {
                $mergedProducts[] = $product;
            } else {
                $simpleProducts[] = $product;
            }
        }

        $logger->notice('AI Enhance: Processing @merged merged and @simple simple products.', [
            '@merged' => count($mergedProducts),
            '@simple' => count($simpleProducts),
        ]);

        // Process merged products first (each as its own batch).
        foreach ($mergedProducts as $product) {
            $this->processMergedProduct($product);
        }

        // Process simple products in regular batches.
        $this->processSimpleProducts($simpleProducts);
    }

    /**
     * Categorizes a product as 'merged' or 'simple'.
     *
     * @param \Drupal\commerce_product\Entity\ProductInterface $product
     *   The commerce product.
     *
     * @return string
     *   'merged' if any variation has multiple sources, 'simple' otherwise.
     */
    protected function categorizeProduct(ProductInterface $product): string
    {
        foreach ($product->getVariations() as $variation) {
            if (!$variation->hasField('field_source_supplier_products')) {
                continue;
            }

            $sources = $variation->get('field_source_supplier_products')->referencedEntities();
            if (count($sources) > 1) {
                return 'merged';
            }
        }
        return 'simple';
    }

    /**
     * Processes a merged product (all sources together in one batch).
     *
     * @param \Drupal\commerce_product\Entity\ProductInterface $product
     *   The merged commerce product.
     */
    protected function processMergedProduct(ProductInterface $product): void
    {
        $logger = $this->loggerFactory->get('sogan_commerce_product');
        $aiConfig = $this->configFactory->getEditable('supplier_products_ai_rewrite.settings');

        // Collect all unique supplier products from all variations.
        $supplierProducts = $this->collectSupplierProducts($product);

        if (empty($supplierProducts)) {
            $logger->warning('Merged product @id has no supplier products.', ['@id' => $product->id()]);
            return;
        }

        $logger->notice('Processing merged product @id with @count supplier sources.', [
            '@id' => $product->id(),
            '@count' => count($supplierProducts),
        ]);

        // Save original context and append merge instructions.
        $originalContext = $aiConfig->get('context_instructions') ?? '';
        $mergeInstructions = $aiConfig->get('merge_instructions') ?? '';

        if (!empty($mergeInstructions)) {
            $combinedContext = $originalContext . "\n\n" . $mergeInstructions;
            $aiConfig->set('context_instructions', $combinedContext)->save();
        }

        try {
            // Get the queue worker instance.
            $queueWorker = $this->queueWorkerManager->createInstance('ai_content_enhancer');

            // Determine tasks to process.
            $tasks = $this->getEnabledTasks($aiConfig);

            // Process all supplier products (bypassing batch size for merged products).
            foreach ($supplierProducts as $supplierProduct) {
                $queueWorker->processItem([
                    'nid' => $supplierProduct->id(),
                    'tasks' => $tasks,
                ]);
            }

            // Flush the batch immediately.
            if (class_exists('\Drupal\supplier_products_ai_rewrite\Plugin\QueueWorker\AIContentEnhancer')) {
                \Drupal\supplier_products_ai_rewrite\Plugin\QueueWorker\AIContentEnhancer::flushBatch();
            }
        } catch (\Exception $e) {
            $logger->error('AI enhancement failed for merged product @id: @message', [
                '@id' => $product->id(),
                '@message' => $e->getMessage(),
            ]);
        }

        // Restore original context instructions.
        if (!empty($mergeInstructions)) {
            $aiConfig->set('context_instructions', $originalContext)->save();
        }

        // Reload supplier products to get AI-enhanced fields.
        $this->entityTypeManager->getStorage('node')->resetCache(array_keys($supplierProducts));

        // Recalculate merged fields on the product.
        $this->recalculateProduct($product);

        $this->messenger->addStatus($this->t('AI enhanced merged product "@title" (@count sources).', [
            '@title' => $product->label(),
            '@count' => count($supplierProducts),
        ]));
    }

    /**
     * Processes simple products in regular batches.
     *
     * @param array $products
     *   Array of simple commerce products.
     */
    protected function processSimpleProducts(array $products): void
    {
        if (empty($products)) {
            return;
        }

        $logger = $this->loggerFactory->get('sogan_commerce_product');
        $aiConfig = $this->configFactory->get('supplier_products_ai_rewrite.settings');
        $batchSize = (int) ($aiConfig->get('batch_size') ?? 5);
        $tasks = $this->getEnabledTasks($aiConfig);

        // Collect all supplier products with their parent commerce product.
        $supplierToProduct = [];
        $allSupplierProducts = [];

        foreach ($products as $product) {
            $sources = $this->collectSupplierProducts($product);
            foreach ($sources as $spId => $supplierProduct) {
                $supplierToProduct[$spId] = $product;
                $allSupplierProducts[$spId] = $supplierProduct;
            }
        }

        if (empty($allSupplierProducts)) {
            return;
        }

        $logger->notice('Processing @count simple products with @sp supplier sources.', [
            '@count' => count($products),
            '@sp' => count($allSupplierProducts),
        ]);

        try {
            $queueWorker = $this->queueWorkerManager->createInstance('ai_content_enhancer');
            $processedCount = 0;
            $productIdsToRecalculate = [];

            foreach ($allSupplierProducts as $spId => $supplierProduct) {
                $queueWorker->processItem([
                    'nid' => $supplierProduct->id(),
                    'tasks' => $tasks,
                ]);
                $processedCount++;
                $productIdsToRecalculate[$supplierToProduct[$spId]->id()] = $supplierToProduct[$spId];

                // Flush batch and recalculate when batch is full.
                if ($processedCount % $batchSize === 0) {
                    if (class_exists('\Drupal\supplier_products_ai_rewrite\Plugin\QueueWorker\AIContentEnhancer')) {
                        \Drupal\supplier_products_ai_rewrite\Plugin\QueueWorker\AIContentEnhancer::flushBatch();
                    }

                    // Recalculate all affected products.
                    foreach ($productIdsToRecalculate as $commerceProduct) {
                        $this->recalculateProduct($commerceProduct);
                    }
                    $productIdsToRecalculate = [];
                }
            }

            // Flush any remaining items.
            if (class_exists('\Drupal\supplier_products_ai_rewrite\Plugin\QueueWorker\AIContentEnhancer')) {
                \Drupal\supplier_products_ai_rewrite\Plugin\QueueWorker\AIContentEnhancer::flushBatch();
            }

            // Recalculate remaining products.
            foreach ($productIdsToRecalculate as $commerceProduct) {
                $this->recalculateProduct($commerceProduct);
            }
        } catch (\Exception $e) {
            $logger->error('AI enhancement failed for simple products: @message', [
                '@message' => $e->getMessage(),
            ]);
        }

        $this->messenger->addStatus($this->t('AI enhanced @count simple products.', [
            '@count' => count($products),
        ]));
    }

    /**
     * Collects all unique supplier products from a commerce product's variations.
     *
     * @param \Drupal\commerce_product\Entity\ProductInterface $product
     *   The commerce product.
     *
     * @return \Drupal\node\NodeInterface[]
     *   Array of supplier product nodes, keyed by node ID.
     */
    protected function collectSupplierProducts(ProductInterface $product): array
    {
        $supplierProducts = [];

        foreach ($product->getVariations() as $variation) {
            if (!$variation->hasField('field_source_supplier_products')) {
                continue;
            }

            foreach ($variation->get('field_source_supplier_products')->referencedEntities() as $supplierProduct) {
                if ($supplierProduct instanceof NodeInterface) {
                    $supplierProducts[$supplierProduct->id()] = $supplierProduct;
                }
            }
        }

        return $supplierProducts;
    }

    /**
     * Gets enabled tasks from config.
     *
     * @param \Drupal\Core\Config\ImmutableConfig $config
     *   The AI config.
     *
     * @return array
     *   Array of enabled task names.
     */
    protected function getEnabledTasks($config): array
    {
        $tasks = [];
        if ($config->get('title_enabled')) {
            $tasks[] = 'title';
        }
        if ($config->get('description_enabled')) {
            $tasks[] = 'description';
        }
        if ($config->get('attributes_enabled')) {
            $tasks[] = 'attributes';
        }
        if ($config->get('categories_enabled')) {
            $tasks[] = 'categorize';
        }
        if ($config->get('brand_enabled')) {
            $tasks[] = 'brand';
        }
        return $tasks;
    }

    /**
     * Recalculates merged fields on a product.
     *
     * @param \Drupal\commerce_product\Entity\ProductInterface $product
     *   The commerce product.
     */
    protected function recalculateProduct(ProductInterface $product): void
    {
        try {
            // Reload product to get fresh data.
            $product = $this->entityTypeManager->getStorage('commerce_product')->load($product->id());
            if ($product) {
                $this->productBuilder->recalculateProductFields($product);

                foreach ($product->getVariations() as $variation) {
                    $this->productBuilder->updateVariationWithSuppliers($variation, []);
                }
            }
        } catch (\Exception $e) {
            $this->loggerFactory->get('sogan_commerce_product')->error('Failed to recalculate product @id: @message', [
                '@id' => $product->id(),
                '@message' => $e->getMessage(),
            ]);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE)
    {
        /** @var \Drupal\commerce_product\Entity\ProductInterface $object */
        return $object->access('update', $account, $return_as_object);
    }
}
