<?php

declare(strict_types=1);

namespace Drupal\sogan_commerce_product\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class SkuSettingsForm extends ConfigFormBase
{

    protected EntityTypeManagerInterface $entityTypeManager;

    public function __construct(EntityTypeManagerInterface $entity_type_manager)
    {
        $this->entityTypeManager = $entity_type_manager;
    }

    public static function create(ContainerInterface $container)
    {
        return new static(
            $container->get('entity_type.manager')
        );
    }

    protected function getEditableConfigNames(): array
    {
        return ['sogan_commerce_product.sku_settings'];
    }

    public function getFormId(): string
    {
        return 'sogan_commerce_product_sku_settings_form';
    }

    public function buildForm(array $form, FormStateInterface $form_state): array
    {
        $config = $this->config('sogan_commerce_product.sku_settings');

        $has_variations = (bool) $this->entityTypeManager
            ->getStorage('commerce_product_variation')
            ->getQuery()
            ->accessCheck(FALSE)
            ->range(0, 1)
            ->execute();

        $form['sku_prefix'] = [
            '#type' => 'textfield',
            '#title' => $this->t('SKU Prefix'),
            '#default_value' => $config->get('sku_prefix') ?? 'P-',
            '#description' => $this->t('The prefix for the SKU (e.g., P-).'),
            '#disabled' => $has_variations,
        ];

        $form['id_length'] = [
            '#type' => 'number',
            '#title' => $this->t('ID Length'),
            '#default_value' => $config->get('id_length') ?? 6,
            '#min' => 1,
            '#max' => 20,
            '#description' => $this->t('Total length of the numeric part (e.g., 6 for 000001).'),
            '#disabled' => $has_variations,
        ];

        if ($has_variations) {
            $form['locked_message'] = [
                '#type' => 'markup',
                '#markup' => '<div class="messages messages--warning">' .
                    $this->t('Prefix and Length settings are locked because product variations exist in the database.') .
                    '</div>',
                '#weight' => -10,
            ];
        }

        $form['visibility_mode'] = [
            '#type' => 'radios',
            '#title' => $this->t('SKU Field Visibility & Behavior'),
            '#default_value' => $config->get('visibility_mode') ?? 'readonly',
            '#options' => [
                'do_nothing' => $this->t('Do nothing (Disable module logic)'),
                'optional' => $this->t('Leave field as it is but create SKU if left empty'),
                'readonly' => $this->t('Make field read-only and auto-generate SKU value'),
                'hidden' => $this->t('Hide field and auto-generate SKU value'),
            ],
        ];

        $form['sequence_init_from_existing'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Initialize sequence from existing SKUs'),
            '#default_value' => (bool) ($config->get('sequence_init_from_existing') ?? FALSE),
            '#description' => $this->t('When enabled, when a new SKU prefix is used, the sequence will initialize from the highest matching existing numeric SKU for that prefix (useful for migrations/imports).'),
        ];

        return parent::buildForm($form, $form_state);
    }

    public function submitForm(array &$form, FormStateInterface $form_state): void
    {
        $this->config('sogan_commerce_product.sku_settings')
            ->set('sku_prefix', $form_state->getValue('sku_prefix'))
            ->set('id_length', $form_state->getValue('id_length'))
            ->set('visibility_mode', $form_state->getValue('visibility_mode'))
            ->set('sequence_init_from_existing', $form_state->getValue('sequence_init_from_existing'))
            ->save();
        parent::submitForm($form, $form_state);
    }

    public function validateForm(array &$form, FormStateInterface $form_state): void
    {
        $prefix = $form_state->getValue('sku_prefix');
        if (!preg_match('/^[A-Za-z0-9_\-]*$/', $prefix)) {
            $form_state->setErrorByName('sku_prefix', $this->t('The SKU prefix can only contain letters, numbers, underscore and hyphen.'));
        }
        $length = (int) $form_state->getValue('id_length');
        if ($length <= 0) {
            $form_state->setErrorByName('id_length', $this->t('ID length must be a positive integer.'));
        }
    }
}
