<?php

namespace Drupal\sogan_commerce_product\Form;

use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\sogan_commerce_product\Service\StockManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirmation form for removing a supplier product from a commerce product.
 */
class RemoveSupplierProductForm extends ConfirmFormBase
{
  /**
   * The product ID.
   *
   * @var int
   */
  protected $productId;

  /**
   * The supplier product ID.
   *
   * @var int
   */
  protected $supplierProductId;

  /**
   * The product builder service.
   *
   * @var \Drupal\sogan_commerce_product\Service\ProductBuilder
   */
  protected $productBuilder;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The stock manager service.
   *
   * @var \Drupal\sogan_commerce_product\Service\StockManager
   */
  protected $stockManager;

  /**
   * The cache tags invalidator.
   *
   * @var \Drupal\Core\Cache\CacheTagsInvalidatorInterface
   */
  protected $cacheTagsInvalidator;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    $instance = parent::create($container);
    $instance->productBuilder = $container->get('sogan_commerce_product.product_builder');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->stockManager = $container->get('sogan_commerce_product.stock_manager');
    $instance->cacheTagsInvalidator = $container->get('cache_tags.invalidator');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId()
  {
    return 'remove_supplier_product_confirm';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion()
  {
    return $this->t('Are you sure you want to remove this supplier product?');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription()
  {
    return $this->t('This will remove the supplier product from all variations and recalculate product data (prices, attributes, stock). This action cannot be undone.');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl()
  {
    return new Url('view.commerce_product_management.page_1');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $product_id = NULL, $supplier_product_id = NULL)
  {
    $this->productId = $product_id;
    $this->supplierProductId = $supplier_product_id;

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    // Load the commerce product
    $product_storage = $this->entityTypeManager->getStorage('commerce_product');
    $product = $product_storage->load($this->productId);

    if (!$product) {
      $this->messenger()->addError($this->t('Product not found.'));
      $form_state->setRedirectUrl($this->getCancelUrl());
      return;
    }

    // Load all variations and check which ones will be affected
    $variations = $product->getVariations();
    $variations_to_delete = [];
    $variations_to_update = [];
    $removed_count = 0;

    foreach ($variations as $variation) {
      if ($variation->hasField('field_source_supplier_products')) {
        $current_refs = $variation->get('field_source_supplier_products')->getValue();
        $has_supplier = FALSE;
        $will_be_empty = TRUE;

        // Check if this variation has the supplier product to remove
        foreach ($current_refs as $ref) {
          if ($ref['target_id'] == $this->supplierProductId) {
            $has_supplier = TRUE;
          } else {
            $will_be_empty = FALSE; // There are other supplier products
          }
        }

        if ($has_supplier) {
          $removed_count++;

          if ($will_be_empty) {
            // This variation will have no supplier products left - delete it
            $variations_to_delete[] = $variation;
          } else {
            // This variation has other supplier products - just update the reference
            $variations_to_update[] = $variation;
          }
        }
      }
    }

    // Delete variations that will be empty
    foreach ($variations_to_delete as $variation) {
      $variation->delete();
    }

    // Update variations that still have other supplier products
    foreach ($variations_to_update as $variation) {
      $current_refs = $variation->get('field_source_supplier_products')->getValue();
      $updated_refs = [];

      foreach ($current_refs as $ref) {
        if ($ref['target_id'] != $this->supplierProductId) {
          $updated_refs[] = $ref;
        }
      }

      $variation->set('field_source_supplier_products', $updated_refs);
      $variation->save();

      // Recalculate prices for this variation
      $this->recalculateVariationPrices($variation);
    }

    // Check if product still has variations
    $product = $product_storage->load($this->productId); // Reload to get fresh data
    $remaining_variations = $product->getVariations();

    if (empty($remaining_variations)) {
      // No variations left - delete the product
      $product_title = $product->getTitle();
      $product->delete();
      $this->messenger()->addWarning($this->t('Product "@title" was deleted because it had no remaining variations after removing the supplier product.', [
        '@title' => $product_title,
      ]));
    } else {
      // Recalculate product-level data
      $this->recalculateProductData($product);

      $delete_msg = count($variations_to_delete) > 0 ? $this->t(', deleted @deleted variation(s)', ['@deleted' => count($variations_to_delete)]) : '';
      $update_msg = count($variations_to_update) > 0 ? $this->t(', updated @updated variation(s)', ['@updated' => count($variations_to_update)]) : '';

      $this->messenger()->addStatus($this->t('Successfully removed supplier product from @count variation(s)@delete@update and recalculated product data.', [
        '@count' => $removed_count,
        '@delete' => $delete_msg,
        '@update' => $update_msg,
      ]));
    }

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

  /**
   * Recalculate prices for a single variation.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
   *   The variation entity.
   */
  /**
   * Recalculate prices for a single variation.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
   *   The variation entity.
   */
  protected function recalculateVariationPrices($variation)
  {
    if (!$variation->hasField('field_source_supplier_products')) {
      return;
    }

    $remaining_suppliers = $variation->get('field_source_supplier_products')->referencedEntities();

    if (empty($remaining_suppliers)) {
      return;
    }

    // Use ProductBuilder to update variation with remaining suppliers
    // This will apply all configured merge strategies (prices, stock, etc.)
    $this->productBuilder->updateVariationWithSuppliers($variation, $remaining_suppliers);
  }

  /**
   * Recalculate stock levels for a variation.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
   *   The variation entity.
   * @param float $new_stock_level
   *   The new total stock level from remaining suppliers.
   */
  protected function recalculateStock($variation, $new_stock_level)
  {
    // Get the stock manager service
    $stock_manager = $this->stockManager;

    // Get stock location from variation's supplier
    $stock_location = NULL;

    if ($variation->hasField('field_source_supplier_products')) {
      $suppliers = $variation->get('field_source_supplier_products')->referencedEntities();

      if (!empty($suppliers)) {
        $first_supplier = reset($suppliers);

        // Get supplier taxonomy term
        if ($first_supplier->hasField('field_supplier') && !$first_supplier->get('field_supplier')->isEmpty()) {
          $supplier_tid = $first_supplier->get('field_supplier')->target_id;

          if ($supplier_tid) {
            $supplier_term = $this->entityTypeManager->getStorage('taxonomy_term')->load($supplier_tid);

            if ($supplier_term && $supplier_term->hasField('field_stock_location') && !$supplier_term->get('field_stock_location')->isEmpty()) {
              $stock_location_id = $supplier_term->get('field_stock_location')->target_id;
              $stock_location = $this->entityTypeManager->getStorage('commerce_stock_location')->load($stock_location_id);
            }
          }
        }
      }
    }

    // If no stock location from field, try to find by supplier name match
    if (!$stock_location && !empty($suppliers)) {
      $first_supplier = reset($suppliers);
      if ($first_supplier->hasField('field_supplier') && !$first_supplier->get('field_supplier')->isEmpty()) {
        $supplier_tid = $first_supplier->get('field_supplier')->target_id;
        if ($supplier_tid) {
          $supplier_term = $this->entityTypeManager->getStorage('taxonomy_term')->load($supplier_tid);
          if ($supplier_term) {
            $supplier_name = $supplier_term->getName();

            // Try to find stock location by name
            $location_storage = $this->entityTypeManager->getStorage('commerce_stock_location');
            $locations = $location_storage->loadByProperties(['name' => $supplier_name]);

            if (!empty($locations)) {
              $stock_location = reset($locations);
            }
          }
        }
      }
    }

    if ($stock_location) {
      // Update stock level using the stock manager
      $stock_manager->setStockLevel($variation, $stock_location, $new_stock_level);
    }
  }

  /**
   * Recalculate product data after removing a supplier product.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $product
   *   The product entity.
   */
  protected function recalculateProductData($product)
  {
    // Recalculate product fields (title, description) based on remaining suppliers
    $this->productBuilder->recalculateProductFields($product);

    // Clear product cache to ensure views display fresh data
    $this->cacheTagsInvalidator->invalidateTags([
      'commerce_product:' . $product->id(),
      'commerce_product_variation_list',
    ]);
  }
}
