<?php

namespace Drupal\sogan_commerce_product\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure merge policy settings for product creation.
 */
class MergeSettingsForm extends ConfigFormBase
{
  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames()
  {
    return ['sogan_commerce_product.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId()
  {
    return 'sogan_commerce_product_merge_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state)
  {
    $config = $this->config('sogan_commerce_product.settings');

    $form['description'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('Configure how product data is merged when creating commerce products from multiple supplier products.') . '</p>',
    ];

    // Text field strategies
    $form['text_fields'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Text Field Strategies'),
      '#description' => $this->t('Choose how to select values when merging text fields from multiple supplier products.'),
    ];

    $text_options = [
      'oldest' => $this->t('Oldest - Use value from oldest supplier product'),
      'newest' => $this->t('Newest - Use value from newest supplier product'),
      'longest' => $this->t('Longest - Use longest text value'),
      'shortest' => $this->t('Shortest - Use shortest text value'),
      'weight' => $this->t('Weight - Use value from highest priority supplier'),
    ];

    // Allow other modules to alter text strategy options
    \Drupal::moduleHandler()->alter('sogan_commerce_text_strategies', $text_options);

    $form['text_fields']['product_name_strategy'] = [
      '#type' => 'select',
      '#title' => $this->t('Product Name'),
      '#options' => $text_options,
      '#default_value' => $config->get('merge_policy.product_name_strategy') ?? 'longest',
      '#description' => $this->t('Strategy for selecting product name.'),
    ];

    $form['text_fields']['product_description_strategy'] = [
      '#type' => 'select',
      '#title' => $this->t('Product Description'),
      '#options' => $text_options,
      '#default_value' => $config->get('merge_policy.product_description_strategy') ?? 'longest',
      '#description' => $this->t('Strategy for selecting product description.'),
    ];

    // Image field strategies
    $form['image_fields'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Image Strategies'),
      '#description' => $this->t('Choose how to merge images from multiple supplier products.'),
    ];

    $image_options = [
      'combine' => $this->t('Combine all images - Merge images from all suppliers'),
      'most_images' => $this->t('Most images - Use images from the supplier with the most images'),
    ];

    $form['image_fields']['product_image_strategy'] = [
      '#type' => 'select',
      '#title' => $this->t('Product Images'),
      '#options' => $image_options,
      '#default_value' => $config->get('merge_policy.product_image_strategy') ?? 'combine',
      '#description' => $this->t('Strategy for selecting product images.'),
    ];

    // Price field strategies
    $form['price_fields'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Price Field Strategies'),
      '#description' => $this->t('Choose how to calculate prices when merging multiple supplier products.'),
    ];

    $price_options = [
      'minimum' => $this->t('Minimum - Use lowest price'),
      'maximum' => $this->t('Maximum - Use highest price'),
      'average' => $this->t('Average - Calculate average price'),
      'weighted' => $this->t('Weighted - Calculate weighted average by stock'),
    ];

    // Allow other modules to alter price strategy options
    \Drupal::moduleHandler()->alter('sogan_commerce_price_strategies', $price_options);

    $form['price_fields']['cost_price_strategy'] = [
      '#type' => 'select',
      '#title' => $this->t('Cost Price'),
      '#options' => $price_options,
      '#default_value' => $config->get('merge_policy.cost_price_strategy') ?? 'weighted',
      '#description' => $this->t('Strategy for calculating cost price (field_cost_price).'),
    ];

    $form['price_fields']['list_price_strategy'] = [
      '#type' => 'select',
      '#title' => $this->t('List Price'),
      '#options' => $price_options,
      '#default_value' => $config->get('merge_policy.list_price_strategy') ?? 'maximum',
      '#description' => $this->t('Strategy for calculating list price (strikethrough price).'),
    ];

    $form['price_fields']['price_strategy'] = [
      '#type' => 'select',
      '#title' => $this->t('Suggested Price'),
      '#options' => $price_options,
      '#default_value' => $config->get('merge_policy.price_strategy') ?? 'weighted',
      '#description' => $this->t('Strategy for calculating suggested/selling price.'),
    ];

    $form['price_fields']['only_instock'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Only consider in-stock supplier products for price calculations'),
      '#default_value' => $config->get('merge_policy.only_instock') ?? TRUE,
      '#description' => $this->t('When calculating prices, only include supplier products with stock > 0. Out-of-stock items will be ignored in price calculations.'),
    ];

    $form['examples'] = [
      '#type' => 'details',
      '#title' => $this->t('Strategy Examples'),
      '#open' => FALSE,
    ];

    $form['examples']['content'] = [
      '#type' => 'markup',
      '#markup' => '
        <h4>' . $this->t('Text Strategies') . '</h4>
        <ul>
          <li><strong>' . $this->t('Oldest') . ':</strong> ' . $this->t('Uses the title/description from the supplier product created first.') . '</li>
          <li><strong>' . $this->t('Newest') . ':</strong> ' . $this->t('Uses the title/description from the most recently created supplier product.') . '</li>
          <li><strong>' . $this->t('Longest') . ':</strong> ' . $this->t('Uses the longest text (most detailed description).') . '</li>
          <li><strong>' . $this->t('Shortest') . ':</strong> ' . $this->t('Uses the shortest text (most concise title).') . '</li>
          <li><strong>' . $this->t('Weight') . ':</strong> ' . $this->t('Uses value from supplier with highest priority/weight.') . '</li>
        </ul>
        <h4>' . $this->t('Image Strategies') . '</h4>
        <ul>
          <li><strong>' . $this->t('Combine all images') . ':</strong> ' . $this->t('Merges all unique images from all suppliers.') . '</li>
          <li><strong>' . $this->t('Most images') . ':</strong> ' . $this->t('Uses images ONLY from the supplier that has the highest number of images.') . '</li>
        </ul>
        <h4>' . $this->t('Price Strategies') . '</h4>
        <ul>
          <li><strong>' . $this->t('Minimum') . ':</strong> ' . $this->t('Best for competitive pricing - shows lowest available price.') . '</li>
          <li><strong>' . $this->t('Maximum') . ':</strong> ' . $this->t('Best for premium positioning - shows highest price.') . '</li>
          <li><strong>' . $this->t('Average') . ':</strong> ' . $this->t('Calculates simple average of all prices.') . '</li>
          <li><strong>' . $this->t('Weighted') . ':</strong> ' . $this->t('Weighted average based on stock levels (higher stock = more weight).') . '</li>
        </ul>
      ',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    $this->config('sogan_commerce_product.settings')
      ->set('merge_policy.product_name_strategy', $form_state->getValue('product_name_strategy'))
      ->set('merge_policy.product_description_strategy', $form_state->getValue('product_description_strategy'))
      ->set('merge_policy.product_image_strategy', $form_state->getValue('product_image_strategy'))
      ->set('merge_policy.cost_price_strategy', $form_state->getValue('cost_price_strategy'))
      ->set('merge_policy.list_price_strategy', $form_state->getValue('list_price_strategy'))
      ->set('merge_policy.price_strategy', $form_state->getValue('price_strategy'))
      ->set('merge_policy.only_instock', $form_state->getValue('only_instock'))
      ->save();

    parent::submitForm($form, $form_state);
  }
}
