<?php

namespace Drupal\sogan_commerce_product\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for Local Stock buffer settings.
 */
class LocalStockSettingsForm extends ConfigFormBase
{

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container)
    {
        $instance = parent::create($container);
        $instance->entityTypeManager = $container->get('entity_type.manager');
        return $instance;
    }

    /**
     * {@inheritdoc}
     */
    protected function getEditableConfigNames()
    {
        return ['sogan_commerce_product.local_stock_settings'];
    }

    /**
     * {@inheritdoc}
     */
    public function getFormId()
    {
        return 'sogan_commerce_product_local_stock_settings';
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(array $form, FormStateInterface $form_state)
    {
        $config = $this->config('sogan_commerce_product.local_stock_settings');

        // Load all stock locations for the dropdown.
        $location_storage = $this->entityTypeManager->getStorage('commerce_stock_location');
        $locations = $location_storage->loadMultiple();

        $options = ['' => $this->t('- Select a location -')];
        foreach ($locations as $location) {
            $options[$location->id()] = $location->label() . ' (ID: ' . $location->id() . ')';
        }

        $form['buffer_location_id'] = [
            '#type' => 'select',
            '#title' => $this->t('Buffer Stock Location'),
            '#description' => $this->t('Select the stock location to use as the buffer for order transactions. When orders are placed, stock will be deducted from this location. This isolates order deductions from supplier feed imports.'),
            '#options' => $options,
            '#default_value' => $config->get('buffer_location_id') ?? '',
            '#required' => TRUE,
        ];

        $form['info'] = [
            '#type' => 'details',
            '#title' => $this->t('How it works'),
            '#open' => FALSE,
        ];

        $form['info']['description'] = [
            '#markup' => '<p>' . $this->t('The buffer stock location acts as an intermediate layer for order transactions:') . '</p>' .
                '<ul>' .
                '<li>' . $this->t('<strong>Order Placed:</strong> Stock is deducted from the buffer location (can go negative)') . '</li>' .
                '<li>' . $this->t('<strong>Feed Import:</strong> Updates only supplier-specific locations, buffer is unaffected') . '</li>' .
                '<li>' . $this->t('<strong>Order Shipped:</strong> If buffer stock is negative, it resets to 0 (items sourced from suppliers)') . '</li>' .
                '</ul>' .
                '<p>' . $this->t('This prevents feed imports from overwriting sales deductions.') . '</p>',
        ];

        return parent::buildForm($form, $form_state);
    }

    /**
     * {@inheritdoc}
     */
    public function submitForm(array &$form, FormStateInterface $form_state)
    {
        $this->config('sogan_commerce_product.local_stock_settings')
            ->set('buffer_location_id', (int) $form_state->getValue('buffer_location_id'))
            ->save();

        parent::submitForm($form, $form_state);
    }
}
