<?php

declare(strict_types=1);

namespace Drupal\sogan_commerce_product\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

class ImageSettingsForm extends ConfigFormBase
{

    protected function getEditableConfigNames(): array
    {
        return ['sogan_commerce_product.sku_settings'];
    }

    public function getFormId(): string
    {
        return 'sogan_commerce_product_image_settings_form';
    }

    public function buildForm(array $form, FormStateInterface $form_state): array
    {
        $config = $this->config('sogan_commerce_product.sku_settings');

        $form['download_remote_images'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Download remote images and create media entities?'),
            '#default_value' => (bool) ($config->get('download_remote_images') ?? TRUE),
            '#description' => $this->t('If enabled, supplier remote image URLs will be downloaded and a Media entity will be created for each image.'),
        ];

        $form['convert_formats'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Convert file formats?'),
            '#default_value' => (bool) ($config->get('convert_formats') ?? TRUE),
            '#description' => $this->t('If enabled, downloaded images may be converted to the selected target format for storage.'),
        ];

        // Detect format support
        $format_support = $this->detectFormatSupport();

        // Build options array with support information
        $format_options = [
            'jpeg' => $this->t('JPEG'),
            'png' => $this->t('PNG'),
        ];

        // Add WebP and AVIF with support status
        if ($format_support['webp']) {
            $format_options['webp'] = $this->t('WebP');
        } else {
            $format_options['webp'] = $this->t('WebP (Not supported by server)');
        }

        if ($format_support['avif']) {
            $format_options['avif'] = $this->t('AVIF');
        } else {
            $format_options['avif'] = $this->t('AVIF (Not supported by server)');
        }

        $form['target_format'] = [
            '#type' => 'select',
            '#title' => $this->t('Target Format'),
            '#options' => $format_options,
            '#default_value' => $config->get('target_format') ?? 'webp',
            '#description' => $this->t('Target format to convert downloaded images to when conversion is enabled.'),
            '#states' => [
                'visible' => [
                    ':input[name="convert_formats"]' => ['checked' => TRUE],
                ],
            ],
        ];

        // Add disabled attribute to unsupported formats using #after_build
        if (!$format_support['webp'] || !$format_support['avif']) {
            $form['target_format']['#after_build'][] = [$this, 'disableUnsupportedFormats'];
            $form['target_format']['#format_support'] = $format_support;
        }

        // File Organization section
        $form['organization'] = [
            '#type' => 'details',
            '#title' => $this->t('File Organization'),
            '#description' => $this->t('Configure how downloaded media files are organized using token patterns.'),
            '#open' => TRUE,
        ];

        $form['organization']['media_directory_pattern'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Directory Pattern'),
            '#default_value' => $config->get('media_directory_pattern') ?: 'public://product_media/[supplier:field_supplier]',
            '#description' => $this->t('Directory path pattern for storing downloaded media. Use tokens to organize files.'),
            '#maxlength' => 255,
            '#required' => TRUE,
        ];

        $form['organization']['media_filename_pattern'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Filename Pattern'),
            '#default_value' => $config->get('media_filename_pattern') ?: '[title]',
            '#description' => $this->t('Filename pattern for downloaded media. File extension will be added automatically.'),
            '#maxlength' => 255,
            '#required' => TRUE,
        ];

        // Token help section
        $form['organization']['tokens'] = [
            '#type' => 'details',
            '#title' => $this->t('Available Tokens'),
            '#description' => $this->t('You can use the following tokens in directory and filename patterns:'),
        ];

        $tokens_info = [
            '[title]' => $this->t('Product/media title (transliterated and sanitized). Example: "Güneş Gözlüğü" becomes "gunes-gozlugu"'),
            '[date:year]' => $this->t('Current year (4 digits). Example: "2025"'),
            '[date:month]' => $this->t('Current month with leading zero. Example: "01", "12"'),
            '[date:day]' => $this->t('Current day with leading zero. Example: "01", "31"'),
            '[random:hash]' => $this->t('Random 8-character hash for uniqueness. Example: "a3f7b2c1"'),
            '[supplier:field_name]' => $this->t('Supplier product field value. Example: [supplier:field_sku] or [supplier:field_supplier]'),
            '[product:field_name]' => $this->t('Commerce product field value. Example: [product:title] or [product:field_product_brand]'),
        ];

        $tokens_list = '<dl>';
        foreach ($tokens_info as $token => $description) {
            $tokens_list .= '<dt><code>' . htmlspecialchars($token) . '</code></dt>';
            $tokens_list .= '<dd>' . $description . '</dd>';
        }
        $tokens_list .= '</dl>';

        $form['organization']['tokens']['list'] = [
            '#markup' => $tokens_list,
        ];

        // Add available fields listings
        $form['organization']['tokens']['available_fields'] = [
            '#type' => 'details',
            '#title' => $this->t('Available Entity Fields'),
            '#description' => $this->t('The following fields are available in your system for use in tokens:'),
        ];

        // List supplier product fields
        $supplier_fields_markup = $this->buildFieldsListing('node', 'supplier_product', 'supplier');
        if ($supplier_fields_markup) {
            $form['organization']['tokens']['available_fields']['supplier'] = [
                '#type' => 'details',
                '#title' => $this->t('Supplier Product Fields'),
                '#markup' => $supplier_fields_markup,
            ];
        }

        // List commerce product fields
        $product_fields_markup = $this->buildFieldsListing('commerce_product', 'default', 'product');
        if ($product_fields_markup) {
            $form['organization']['tokens']['available_fields']['product'] = [
                '#type' => 'details',
                '#title' => $this->t('Commerce Product Fields'),
                '#markup' => $product_fields_markup,
            ];
        }

        // Example patterns
        $form['organization']['tokens']['examples'] = [
            '#type' => 'details',
            '#title' => $this->t('Pattern Examples'),
        ];

        $examples = [
            [
                'description' => $this->t('Organize by year and month'),
                'directory' => 'public://product_media/[date:year]/[date:month]',
                'filename' => '[title]',
                'result' => 'public://product_media/2025/11/gunes-gozlugu.jpg',
            ],
            [
                'description' => $this->t('Organize by supplier name'),
                'directory' => 'public://product_media/[supplier:field_supplier]',
                'filename' => '[title]',
                'result' => 'public://product_media/tedarikci-adi/gunes-gozlugu.jpg',
            ],
            [
                'description' => $this->t('Add random hash to avoid collisions'),
                'directory' => 'public://product_media',
                'filename' => '[title]-[random:hash]',
                'result' => 'public://product_media/gunes-gozlugu-a3f7b2c1.jpg',
            ],
            [
                'description' => $this->t('Use supplier SKU in filename'),
                'directory' => 'public://product_media/[date:year]/[date:month]',
                'filename' => '[supplier:field_sku]-[title]',
                'result' => 'public://product_media/2025/11/sup123-gunes-gozlugu.jpg',
            ],
        ];

        $examples_markup = '<table><thead><tr><th>' . $this->t('Description') . '</th><th>' . $this->t('Directory Pattern') . '</th><th>' . $this->t('Filename Pattern') . '</th><th>' . $this->t('Example Result') . '</th></tr></thead><tbody>';
        foreach ($examples as $pattern) {
            $examples_markup .= '<tr>';
            $examples_markup .= '<td>' . $pattern['description'] . '</td>';
            $examples_markup .= '<td><code>' . htmlspecialchars($pattern['directory']) . '</code></td>';
            $examples_markup .= '<td><code>' . htmlspecialchars($pattern['filename']) . '</code></td>';
            $examples_markup .= '<td><code>' . htmlspecialchars($pattern['result']) . '</code></td>';
            $examples_markup .= '</tr>';
        }
        $examples_markup .= '</tbody></table>';

        $form['organization']['tokens']['examples']['table'] = [
            '#markup' => $examples_markup,
        ];


        return parent::buildForm($form, $form_state);
    }

    /**
     * After build callback to disable unsupported format options.
     */
    public function disableUnsupportedFormats($element, $form_state)
    {
        $support = $element['#format_support'] ?? [];

        if (isset($element['#options'])) {
            foreach (['webp', 'avif'] as $format) {
                if (!empty($support[$format])) {
                    continue; // Format is supported
                }

                // Disable the option by modifying the render array
                if (isset($element[$format])) {
                    $element[$format]['#disabled'] = TRUE;
                    $element[$format]['#attributes']['disabled'] = 'disabled';
                }
            }
        }

        return $element;
    }

    /**
     * Detect which image formats are supported by the server.
     *
     * @return array
     *   Array with 'webp' and 'avif' boolean support flags.
     */
    protected function detectFormatSupport(): array
    {
        $support = [
            'webp' => FALSE,
            'avif' => FALSE,
        ];

        // Check GD support
        if (function_exists('gd_info')) {
            $gd_info = gd_info();

            // Check for WebP support in GD
            if (!empty($gd_info['WebP Support'])) {
                $support['webp'] = TRUE;
            }

            // Check for AVIF support in GD (PHP 8.1+)
            if (!empty($gd_info['AVIF Support'])) {
                $support['avif'] = TRUE;
            }
        }

        // Check ImageMagick support if GD doesn't support
        if ((!$support['webp'] || !$support['avif']) && extension_loaded('imagick')) {
            try {
                $imagick = new \Imagick();
                $formats = $imagick->queryFormats();

                if (!$support['webp'] && in_array('WEBP', $formats)) {
                    $support['webp'] = TRUE;
                }

                if (!$support['avif'] && in_array('AVIF', $formats)) {
                    $support['avif'] = TRUE;
                }
            } catch (\Throwable $e) {
                // ImageMagick detection failed, keep defaults
            }
        }

        return $support;
    }

    /**
     * Build a markup list of available fields for an entity type/bundle.
     *
     * @param string $entity_type
     *   The entity type ID.
     * @param string $bundle
     *   The bundle name.
     * @param string $token_prefix
     *   The token prefix (e.g., 'supplier' or 'product').
     *
     * @return string|null
     *   HTML markup with field listing or NULL if no fields found.
     */
    protected function buildFieldsListing(string $entity_type, string $bundle, string $token_prefix): ?string
    {
        try {
            $entity_field_manager = \Drupal::service('entity_field.manager');
            $field_definitions = $entity_field_manager->getFieldDefinitions($entity_type, $bundle);

            if (empty($field_definitions)) {
                return NULL;
            }

            $fields_data = [];
            foreach ($field_definitions as $field_name => $field_definition) {
                // Skip base fields that don't make sense for file paths
                $skip_fields = ['uuid', 'revision_log', 'revision_uid', 'revision_timestamp', 'langcode', 'default_langcode', 'revision_translation_affected', 'content_translation_source', 'content_translation_outdated', 'content_translation_uid', 'content_translation_status', 'content_translation_created', 'content_translation_changed'];
                if (in_array($field_name, $skip_fields)) {
                    continue;
                }

                $field_type = $field_definition->getType();
                $field_label = (string) $field_definition->getLabel();

                // Only include simple field types that can be used in paths
                $allowed_types = ['string', 'string_long', 'text', 'text_long', 'text_with_summary', 'integer', 'decimal', 'float', 'entity_reference', 'list_string', 'list_integer', 'datetime', 'timestamp'];
                if (in_array($field_type, $allowed_types)) {
                    $fields_data[] = [
                        'name' => $field_name,
                        'label' => $field_label,
                        'type' => $field_type,
                        'token' => '[' . $token_prefix . ':' . $field_name . ']',
                    ];
                }
            }

            if (empty($fields_data)) {
                return NULL;
            }

            // Build table markup
            $markup = '<table class="token-fields-table">';
            $markup .= '<thead><tr>';
            $markup .= '<th>' . $this->t('Token') . '</th>';
            $markup .= '<th>' . $this->t('Field Label') . '</th>';
            $markup .= '<th>' . $this->t('Field Type') . '</th>';
            $markup .= '</tr></thead><tbody>';

            foreach ($fields_data as $field) {
                $markup .= '<tr>';
                $markup .= '<td><code>' . htmlspecialchars($field['token']) . '</code></td>';
                $markup .= '<td>' . htmlspecialchars($field['label']) . '</td>';
                $markup .= '<td><small>' . htmlspecialchars($field['type']) . '</small></td>';
                $markup .= '</tr>';
            }

            $markup .= '</tbody></table>';

            return $markup;
        } catch (\Throwable $e) {
            \Drupal::logger('sogan_commerce_product')->error('Failed to build fields listing: @msg', ['@msg' => $e->getMessage()]);
            return NULL;
        }
    }

    public function submitForm(array &$form, FormStateInterface $form_state): void
    {
        $this->config('sogan_commerce_product.sku_settings')
            ->set('download_remote_images', (bool) $form_state->getValue('download_remote_images'))
            ->set('convert_formats', (bool) $form_state->getValue('convert_formats'))
            ->set('target_format', $form_state->getValue('target_format'))
            ->set('media_directory_pattern', trim($form_state->getValue('media_directory_pattern')))
            ->set('media_filename_pattern', trim($form_state->getValue('media_filename_pattern')))
            ->save();
        parent::submitForm($form, $form_state);
    }
}
