<?php

namespace Drupal\sogan_commerce_product\EventSubscriber;

use Drupal\commerce_stock\StockTransactionsInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\feeds\Event\FeedsEvents;
use Drupal\feeds\Event\ProcessEntityEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Drupal\commerce_stock\StockServiceManager;
use Drupal\sogan_commerce_product\Service\StockManager;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Synchronizes stock data from feeds to the associated product variation.
 */
class StockSyncSubscriber implements EventSubscriberInterface
{

    protected $stockServiceManager;
    protected $entityTypeManager;
    protected $stockManager;

    /**
     * The logger channel.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    public function __construct(StockServiceManager $stock_service_manager, EntityTypeManagerInterface $entity_type_manager, StockManager $stock_manager, LoggerChannelFactoryInterface $logger_factory)
    {
        $this->stockServiceManager = $stock_service_manager;
        $this->entityTypeManager = $entity_type_manager;
        $this->stockManager = $stock_manager;
        $this->logger = $logger_factory->get('sogan_commerce_product');
    }

    /**
     * {@inheritdoc}
     */
    public static function getSubscribedEvents()
    {
        // Using string for compatibility with different Feeds versions.
        return [
            'feeds.process_entity_post_save' => 'onPostSave',
        ];
    }

    /**
     * Logic for stock synchronization.
     */
    public function onPostSave(ProcessEntityEvent $event)
    {
        $entity = $event->getEntity();

        // 1. Only run for 'supplier_product' content type.
        if ($entity->getEntityTypeId() !== 'node' || $entity->bundle() !== 'supplier_product') {
            return;
        }

        // 2. Find any variations that reference this supplier_product node via
        // field_source_supplier_products. This handles the case where multiple
        // variations reference the same supplier product.
        $variation_storage = $this->entityTypeManager->getStorage('commerce_product_variation');
        $query = $variation_storage->getQuery()
            ->condition('field_source_supplier_products', $entity->id())
            ->accessCheck(FALSE)
            ->range(0, 50);

        $variation_ids = $query->execute();
        if (empty($variation_ids)) {
            return;
        }

        $variations = $variation_storage->loadMultiple($variation_ids);

        // 3. Determine Which Supplier? (Stock Location Detection)
        if (!$entity->hasField('field_supplier') || $entity->get('field_supplier')->isEmpty()) {
            return;
        }
        $supplier_term = $entity->get('field_supplier')->entity;

        // Get the Stock Location ID linked to the Supplier.
        if (!$supplier_term || !$supplier_term->hasField('field_stock_location') || $supplier_term->get('field_stock_location')->isEmpty()) {
            // If the supplier doesn't have a location, we can't process stock.
            return;
        }
        // Build aggregated stock per variation / per supplier for this supplier_term.
        foreach ($variations as $variation) {
            // Find all supplier product nodes referenced by this variation that
            // belong to the same supplier term as the updated node, and sum
            // their stock values.
            if (!$variation->hasField('field_source_supplier_products') || $variation->get('field_source_supplier_products')->isEmpty()) {
                continue;
            }
            $referenced = $variation->get('field_source_supplier_products')->referencedEntities();
            $sum = 0.0;
            foreach ($referenced as $ref_node) {
                if (!$ref_node->hasField('field_supplier') || $ref_node->get('field_supplier')->isEmpty()) {
                    continue;
                }
                $term = $ref_node->get('field_supplier')->entity;
                if ($term && $term->id() === $supplier_term->id()) {
                    // Treat negative supplier stock as zero when aggregating
                    // to avoid decreasing the location-level total.
                    $val = (float) $this->stockManager->getSupplierStock($ref_node);
                    if ($val < 0.0) {
                        $val = 0.0;
                    }
                    $sum += $val;
                }
            }

            // Set location-level stock equal to the aggregated sum.
            $location = $this->stockManager->getSupplierLocation($entity);
            if (!$location) {
                $this->logger->warning('No stock location for supplier when syncing update for supplier_product @nid', ['@nid' => $entity->id()]);
                continue;
            }

            $message = 'Auto-sync aggregated supplier stock after supplier_product update (source node: ' . $entity->id() . ')';
            $this->stockManager->setStockLevel($variation, $location, (float) $sum, $message);
        }
    }
}
