<?php

namespace Drupal\sogan_commerce_product\EventSubscriber;

use Drupal\commerce_stock\StockServiceManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\sogan_commerce_product\Service\LocalStockService;
use Drupal\sogan_commerce_product\Service\StockManager;
use Drupal\state_machine\Event\WorkflowTransitionEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Resets negative Local Stock after order shipment/fulfillment.
 *
 * When an order is shipped (fulfilled), if Local Stock is negative it means
 * items were sourced from suppliers. This subscriber resets the negative
 * balance to zero, signifying the sourcing is complete.
 *
 * If Local Stock is positive, it means items came from local inventory
 * and no reset is needed.
 */
class ShipmentStockSubscriber implements EventSubscriberInterface
{

    /**
     * The Local Stock service.
     *
     * @var \Drupal\sogan_commerce_product\Service\LocalStockService
     */
    protected $localStockService;

    /**
     * The stock manager.
     *
     * @var \Drupal\sogan_commerce_product\Service\StockManager
     */
    protected $stockManager;

    /**
     * The stock service manager.
     *
     * @var \Drupal\commerce_stock\StockServiceManagerInterface
     */
    protected $stockServiceManager;

    /**
     * The logger channel.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * Constructs a new ShipmentStockSubscriber.
     *
     * @param \Drupal\sogan_commerce_product\Service\LocalStockService $local_stock_service
     *   The Local Stock service.
     * @param \Drupal\sogan_commerce_product\Service\StockManager $stock_manager
     *   The stock manager.
     * @param \Drupal\commerce_stock\StockServiceManagerInterface $stock_service_manager
     *   The stock service manager.
     * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
     *   The logger factory.
     */
    public function __construct(
        LocalStockService $local_stock_service,
        StockManager $stock_manager,
        StockServiceManagerInterface $stock_service_manager,
        LoggerChannelFactoryInterface $logger_factory
    ) {
        $this->localStockService = $local_stock_service;
        $this->stockManager = $stock_manager;
        $this->stockServiceManager = $stock_service_manager;
        $this->logger = $logger_factory->get('sogan_commerce_product');
    }

    /**
     * {@inheritdoc}
     */
    public static function getSubscribedEvents(): array
    {
        return [
            // Listen for fulfill transition (standard Commerce shipment).
            'commerce_order.fulfill.post_transition' => ['onOrderFulfill', 0],
            // Also listen for complete transition in case workflow uses that.
            'commerce_order.complete.post_transition' => ['onOrderComplete', 0],
        ];
    }

    /**
     * Resets negative Local Stock when order is fulfilled.
     *
     * @param \Drupal\state_machine\Event\WorkflowTransitionEvent $event
     *   The workflow transition event.
     */
    public function onOrderFulfill(WorkflowTransitionEvent $event): void
    {
        $this->resetLocalStockForOrder($event);
    }

    /**
     * Resets negative Local Stock when order is completed.
     *
     * This handles workflows where "complete" is used for shipment.
     *
     * @param \Drupal\state_machine\Event\WorkflowTransitionEvent $event
     *   The workflow transition event.
     */
    public function onOrderComplete(WorkflowTransitionEvent $event): void
    {
        // Only process if transitioning TO completed from fulfillment.
        // This prevents double-processing if fulfill already handled it.
        $transition = $event->getTransition();
        $from_states = $transition->getFromStates();
        $from_state_ids = array_map(function ($state) {
            return $state->getId();
        }, $from_states);

        if (!in_array('fulfillment', $from_state_ids)) {
            return;
        }

        $this->resetLocalStockForOrder($event);
    }

    /**
     * Resets negative Local Stock for all items in an order.
     *
     * @param \Drupal\state_machine\Event\WorkflowTransitionEvent $event
     *   The workflow transition event.
     */
    protected function resetLocalStockForOrder(WorkflowTransitionEvent $event): void
    {
        /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
        $order = $event->getEntity();

        // Only handle commerce orders.
        if ($order->getState()->getWorkflow()->getGroup() !== 'commerce_order') {
            return;
        }

        $items_reset = 0;
        $items_skipped = 0;

        foreach ($order->getItems() as $item) {
            $entity = $item->getPurchasedEntity();
            if (!$entity) {
                continue;
            }

            // Check if always in stock.
            if ($this->isAlwaysInStock($entity)) {
                continue;
            }

            $message = sprintf(
                'Order #%d shipped - resetting Local Stock (items sourced from suppliers)',
                $order->id()
            );

            $result = $this->localStockService->resetNegativeStockAfterShipment($entity, $message);

            if ($result === NULL) {
                // No reset needed (stock was positive).
                $items_skipped++;
            } elseif ($result !== FALSE) {
                $items_reset++;

                $this->logger->info(
                    'Order #@order shipped: Reset negative Local Stock for variation @vid',
                    ['@order' => $order->id(), '@vid' => $entity->id()]
                );
            }
        }

        if ($items_reset > 0) {
            $this->logger->info(
                'Order #@order shipment: Reset Local Stock for @count items (@skipped had positive stock)',
                ['@order' => $order->id(), '@count' => $items_reset, '@skipped' => $items_skipped]
            );
        }
    }

    /**
     * Check if an entity is marked as always in stock.
     *
     * @param \Drupal\commerce\PurchasableEntityInterface $entity
     *   The purchasable entity.
     *
     * @return bool
     *   TRUE if always in stock.
     */
    protected function isAlwaysInStock($entity): bool
    {
        try {
            $service = $this->stockServiceManager->getService($entity);
            $checker = $service->getStockChecker();
            return $checker->getIsAlwaysInStock($entity);
        } catch (\Exception $e) {
            return FALSE;
        }
    }
}
