<?php

namespace Drupal\sogan_commerce_product\EventSubscriber;

use Drupal\commerce_order\Entity\Order;
use Drupal\commerce_order\Event\OrderEvent;
use Drupal\commerce_order\Event\OrderEvents;
use Drupal\commerce_order\Event\OrderItemEvent;
use Drupal\commerce_stock\StockServiceManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\sogan_commerce_product\Service\LocalStockService;
use Drupal\sogan_commerce_product\Service\StockManager;
use Drupal\state_machine\Event\WorkflowTransitionEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Redirects order stock transactions to Local Stock location.
 *
 * This subscriber has higher priority than the default Commerce Stock
 * OrderEventSubscriber and handles all order-related stock transactions
 * by directing them to the Local Stock location instead of supplier locations.
 *
 * This prevents feed imports from overwriting sales deductions.
 */
class OrderStockSubscriber implements EventSubscriberInterface
{

    /**
     * The Local Stock service.
     *
     * @var \Drupal\sogan_commerce_product\Service\LocalStockService
     */
    protected $localStockService;

    /**
     * The stock manager.
     *
     * @var \Drupal\sogan_commerce_product\Service\StockManager
     */
    protected $stockManager;

    /**
     * The stock service manager.
     *
     * @var \Drupal\commerce_stock\StockServiceManagerInterface
     */
    protected $stockServiceManager;

    /**
     * The logger channel.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * Constructs a new OrderStockSubscriber.
     *
     * @param \Drupal\sogan_commerce_product\Service\LocalStockService $local_stock_service
     *   The Local Stock service.
     * @param \Drupal\sogan_commerce_product\Service\StockManager $stock_manager
     *   The stock manager.
     * @param \Drupal\commerce_stock\StockServiceManagerInterface $stock_service_manager
     *   The stock service manager.
     * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
     *   The logger factory.
     */
    public function __construct(
        LocalStockService $local_stock_service,
        StockManager $stock_manager,
        StockServiceManagerInterface $stock_service_manager,
        LoggerChannelFactoryInterface $logger_factory
    ) {
        $this->localStockService = $local_stock_service;
        $this->stockManager = $stock_manager;
        $this->stockServiceManager = $stock_service_manager;
        $this->logger = $logger_factory->get('sogan_commerce_product');
    }

    /**
     * {@inheritdoc}
     */
    public static function getSubscribedEvents(): array
    {
        // Higher priority than Commerce Stock's OrderEventSubscriber (-100).
        // We handle stock transactions ourselves and stop propagation.
        return [
            'commerce_order.place.post_transition' => ['onOrderPlace', 200],
            'commerce_order.cancel.post_transition' => ['onOrderCancel', 200],
            OrderEvents::ORDER_UPDATE => ['onOrderUpdate', 200],
            OrderEvents::ORDER_ITEM_UPDATE => ['onOrderItemUpdate', 200],
            OrderEvents::ORDER_ITEM_DELETE => ['onOrderItemDelete', 200],
        ];
    }

    /**
     * Handles stock deduction when an order is placed.
     *
     * @param \Drupal\state_machine\Event\WorkflowTransitionEvent $event
     *   The workflow transition event.
     */
    public function onOrderPlace(WorkflowTransitionEvent $event): void
    {
        /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
        $order = $event->getEntity();

        // Only handle commerce orders.
        if ($order->getState()->getWorkflow()->getGroup() !== 'commerce_order') {
            return;
        }

        foreach ($order->getItems() as $item) {
            $entity = $item->getPurchasedEntity();
            if (!$entity) {
                continue;
            }

            // Check if always in stock.
            if ($this->isAlwaysInStock($entity)) {
                continue;
            }

            $quantity = $item->getQuantity();
            $message = sprintf('Order #%d placed - deducted from Local Stock', $order->id());

            $this->localStockService->deductStock($entity, $quantity, $message);

            $this->logger->debug(
                'Order #@order: Deducted @qty of variation @vid from Local Stock',
                ['@order' => $order->id(), '@qty' => $quantity, '@vid' => $entity->id()]
            );
        }

        // Stop propagation to prevent Commerce Stock from also handling this.
        $event->stopPropagation();
    }

    /**
     * Handles stock return when an order is canceled.
     *
     * @param \Drupal\state_machine\Event\WorkflowTransitionEvent $event
     *   The workflow transition event.
     */
    public function onOrderCancel(WorkflowTransitionEvent $event): void
    {
        /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
        $order = $event->getEntity();

        // Don't return stock for draft orders (never deducted).
        $original_state = $order->original?->getState()->getId() ?? 'draft';
        if ($original_state === 'draft') {
            return;
        }

        foreach ($order->getItems() as $item) {
            $entity = $item->getPurchasedEntity();
            if (!$entity) {
                continue;
            }

            if ($this->isAlwaysInStock($entity)) {
                continue;
            }

            $quantity = $item->getQuantity();
            $message = sprintf('Order #%d canceled - returned to Local Stock', $order->id());

            $this->localStockService->returnStock($entity, $quantity, $message);

            $this->logger->debug(
                'Order #@order canceled: Returned @qty of variation @vid to Local Stock',
                ['@order' => $order->id(), '@qty' => $quantity, '@vid' => $entity->id()]
            );
        }

        $event->stopPropagation();
    }

    /**
     * Handles stock for new order items added to existing orders.
     *
     * @param \Drupal\commerce_order\Event\OrderEvent $event
     *   The order event.
     */
    public function onOrderUpdate(OrderEvent $event): void
    {
        $order = $event->getOrder();

        // Only handle non-draft orders.
        if (in_array($order->getState()->getId(), ['draft', 'canceled'])) {
            return;
        }

        $original = $order->original;
        if (!$original) {
            return;
        }

        foreach ($order->getItems() as $item) {
            // Check if this is a new item (not in original order).
            if (!$original->hasItem($item)) {
                $entity = $item->getPurchasedEntity();
                if (!$entity || $this->isAlwaysInStock($entity)) {
                    continue;
                }

                $quantity = $item->getQuantity();
                $message = sprintf('Order #%d updated - new item added to Local Stock', $order->id());

                $this->localStockService->deductStock($entity, $quantity, $message);
            }
        }

        // Note: We don't stop propagation here as other subscribers may need
        // to handle non-stock-related order update logic.
    }

    /**
     * Handles stock adjustment when order item quantity changes.
     *
     * @param \Drupal\commerce_order\Event\OrderItemEvent $event
     *   The order item event.
     */
    public function onOrderItemUpdate(OrderItemEvent $event): void
    {
        $item = $event->getOrderItem();
        $order = $item->getOrder();

        if (!$order || in_array($order->getState()->getId(), ['draft', 'canceled'])) {
            return;
        }

        $original = $item->original ?? NULL;
        if (!$original) {
            return;
        }

        $diff = $original->getQuantity() - $item->getQuantity();

        // Only process if quantity changed.
        if (abs($diff) < 0.001) {
            return;
        }

        $entity = $item->getPurchasedEntity();
        if (!$entity || $this->isAlwaysInStock($entity)) {
            return;
        }

        $message = sprintf(
            'Order #%d item updated - quantity changed by %+.2f',
            $order->id(),
            $diff
        );

        if ($diff > 0) {
            // Quantity decreased, return stock.
            $this->localStockService->returnStock($entity, $diff, $message);
        } else {
            // Quantity increased, deduct more stock.
            $this->localStockService->deductStock($entity, abs($diff), $message);
        }
    }

    /**
     * Handles stock return when an order item is deleted.
     *
     * @param \Drupal\commerce_order\Event\OrderItemEvent $event
     *   The order item event.
     */
    public function onOrderItemDelete(OrderItemEvent $event): void
    {
        $item = $event->getOrderItem();
        $order = $item->getOrder();

        if (!$order || in_array($order->getState()->getId(), ['draft', 'canceled'])) {
            return;
        }

        $entity = $item->getPurchasedEntity();
        if (!$entity || $this->isAlwaysInStock($entity)) {
            return;
        }

        $quantity = $item->getQuantity();
        $message = sprintf('Order #%d item deleted - returned to Local Stock', $order->id());

        $this->localStockService->returnStock($entity, $quantity, $message);
    }

    /**
     * Check if an entity is marked as always in stock.
     *
     * @param \Drupal\commerce\PurchasableEntityInterface $entity
     *   The purchasable entity.
     *
     * @return bool
     *   TRUE if always in stock.
     */
    protected function isAlwaysInStock($entity): bool
    {
        try {
            $service = $this->stockServiceManager->getService($entity);
            $checker = $service->getStockChecker();
            return $checker->getIsAlwaysInStock($entity);
        } catch (\Exception $e) {
            return FALSE;
        }
    }
}
