<?php

namespace Drupal\sogan_commerce_product\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseModalDialogCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for managing supplier product relationships.
 */
class SupplierProductController extends ControllerBase
{
  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a SupplierProductController object.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, MessengerInterface $messenger, RequestStack $request_stack)
  {
    $this->entityTypeManager = $entity_type_manager;
    $this->messenger = $messenger;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('messenger'),
      $container->get('request_stack')
    );
  }

  /**
   * Remove a supplier product from a commerce product.
   */
  public function removeSupplierProduct($product_id, $supplier_product_id)
  {
    // Load the commerce product
    $product_storage = $this->entityTypeManager->getStorage('commerce_product');
    $product = $product_storage->load($product_id);

    if (!$product) {
      $this->messenger->addError($this->t('Product not found.'));
      return new RedirectResponse('/admin/commerce/products');
    }

    // Load all variations and remove the supplier product reference
    $variations = $product->getVariations();
    $removed_count = 0;

    foreach ($variations as $variation) {
      if ($variation->hasField('field_source_supplier_products')) {
        $current_refs = $variation->get('field_source_supplier_products')->getValue();
        $updated_refs = [];

        // Filter out the supplier product to remove
        foreach ($current_refs as $ref) {
          if ($ref['target_id'] != $supplier_product_id) {
            $updated_refs[] = $ref;
          } else {
            $removed_count++;
          }
        }

        // Update the variation if changes were made
        if (count($updated_refs) != count($current_refs)) {
          $variation->set('field_source_supplier_products', $updated_refs);
          $variation->save();
        }
      }
    }

    if ($removed_count > 0) {
      $this->messenger->addStatus($this->t('Successfully removed supplier product from @count variation(s).', ['@count' => $removed_count]));
    } else {
      $this->messenger->addWarning($this->t('Supplier product was not found in any variations.'));
    }

    // Check if this is an AJAX request
    $request = $this->requestStack->getCurrentRequest();
    if ($request->query->get('_wrapper_format') === 'drupal_ajax' || $request->isXmlHttpRequest()) {
      $response = new AjaxResponse();
      $response->addCommand(new CloseModalDialogCommand());
      $response->addCommand(new \Drupal\Core\Ajax\InvokeCommand('window', 'location.reload', [TRUE]));
      return $response;
    }

    // Non-AJAX fallback
    return new RedirectResponse('/admin/commerce/products');
  }
}
