<?php

namespace Drupal\sogan_commerce_product\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Controller for testing stock display.
 */
class StockTestController extends ControllerBase
{

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The stock service manager.
   *
   * @var object
   */
  protected $stockServiceManager;

  /**
   * Constructs a StockTestController object.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, $stock_service_manager)
  {
    $this->entityTypeManager = $entity_type_manager;
    $this->stockServiceManager = $stock_service_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('commerce_stock.service_manager')
    );
  }

  /**
   * Test stock display for a product.
   */
  public function testStock($product_id = NULL)
  {
    $product_storage = $this->entityTypeManager->getStorage('commerce_product');

    if ($product_id) {
      $product = $product_storage->load($product_id);
    } else {
      // Get first product
      $query = $product_storage->getQuery()
        ->condition('type', 'default')
        ->range(0, 1)
        ->accessCheck(FALSE);
      $ids = $query->execute();
      if (empty($ids)) {
        return ['#markup' => '<p>' . $this->t('No products found.') . '</p>'];
      }
      $product = $product_storage->load(reset($ids));
    }

    if (!$product) {
      return ['#markup' => '<p>' . $this->t('Product not found.') . '</p>'];
    }

    $output = [];
    $output[] = '<h2>' . $this->t('Product Stock Test') . '</h2>';
    $output[] = '<p><strong>' . $this->t('Product:') . '</strong> ' . $product->getTitle() . ' (' . $this->t('ID:') . ' ' . $product->id() . ')</p>';

    $variations = $product->getVariations();
    $output[] = '<p><strong>' . $this->t('Variations:') . '</strong> ' . count($variations) . '</p>';

    // Get all stock locations
    $location_storage = $this->entityTypeManager->getStorage('commerce_stock_location');
    $all_locations = $location_storage->loadMultiple();

    $output[] = '<p><strong>' . $this->t('Stock Locations:') . '</strong> ' . count($all_locations) . '</p>';
    $output[] = '<ul>';
    foreach ($all_locations as $loc) {
      $output[] = '<li>' . $loc->getName() . ' (' . $this->t('ID:') . ' ' . $loc->id() . ')</li>';
    }
    $output[] = '</ul>';

    // Get stock service manager
    $stock_service_manager = $this->stockServiceManager;

    // Collect stock by location
    $stock_by_location = [];
    foreach ($all_locations as $location) {
      $stock_by_location[$location->getName()] = 0;
    }

    foreach ($variations as $variation) {
      $output[] = '<h3>' . $this->t('Variation ID:') . ' ' . $variation->id() . '</h3>';
      $output[] = '<p><strong>' . $this->t('Title:') . '</strong> ' . $variation->getTitle() . '</p>';

      $stock_service = $stock_service_manager->getService($variation);
      $stock_checker = $stock_service->getStockChecker();

      $output[] = '<p><strong>' . $this->t('Stock Checker:') . '</strong> ' . get_class($stock_checker) . '</p>';

      if (method_exists($stock_checker, 'getLocationsStockLevels')) {
        $output[] = '<p><strong>' . $this->t('Method:') . '</strong> getLocationsStockLevels</p>';
        $levels = $stock_checker->getLocationsStockLevels($variation, $all_locations);
        $output[] = '<table border="1" cellpadding="5">';
        $output[] = '<tr><th>' . $this->t('Location') . '</th><th>' . $this->t('Base Qty') . '</th><th>' . $this->t('Transactions Qty') . '</th><th>' . $this->t('Total') . '</th></tr>';
        foreach ($levels as $loc_id => $data) {
          if (isset($all_locations[$loc_id])) {
            $loc_name = $all_locations[$loc_id]->getName();
            $qty = (float) $data['qty'] + (float) $data['transactions_qty'];
            $stock_by_location[$loc_name] += $qty;
            $output[] = '<tr>';
            $output[] = '<td>' . $loc_name . ' (' . $this->t('ID:') . ' ' . $loc_id . ')</td>';
            $output[] = '<td>' . $data['qty'] . '</td>';
            $output[] = '<td>' . $data['transactions_qty'] . '</td>';
            $output[] = '<td><strong>' . $qty . '</strong></td>';
            $output[] = '</tr>';
          }
        }
        $output[] = '</table>';
      } elseif (method_exists($stock_checker, 'getLocationStockLevel')) {
        $output[] = '<p><strong>' . $this->t('Method:') . '</strong> getLocationStockLevel (' . $this->t('fallback') . ')</p>';
        $output[] = '<table border="1" cellpadding="5">';
        $output[] = '<tr><th>' . $this->t('Location') . '</th><th>' . $this->t('Base Qty') . '</th><th>' . $this->t('Last Txn') . '</th><th>' . $this->t('Latest Txn') . '</th><th>' . $this->t('Transactions Qty') . '</th><th>' . $this->t('Total') . '</th></tr>';
        foreach ($all_locations as $location) {
          $location_id = $location->id();
          $location_name = $location->getName();
          $data = $stock_checker->getLocationStockLevel($location_id, $variation);

          $transactions_qty = 0;
          $latest_txn = 0;
          if (
            method_exists($stock_checker, 'getLocationStockTransactionLatest') &&
            method_exists($stock_checker, 'getLocationStockTransactionSum')
          ) {
            $latest_txn = $stock_checker->getLocationStockTransactionLatest($location_id, $variation);
            $last_transaction = $data['last_transaction'] ?? 0;
            $transactions_qty = $stock_checker->getLocationStockTransactionSum($location_id, $variation, $last_transaction, $latest_txn);
          }

          $base_qty = isset($data['qty']) ? (float) $data['qty'] : 0;
          $qty = $base_qty + (float) $transactions_qty;
          $stock_by_location[$location_name] += $qty;

          $output[] = '<tr>';
          $output[] = '<td>' . $location_name . ' (' . $this->t('ID:') . ' ' . $location_id . ')</td>';
          $output[] = '<td>' . $base_qty . '</td>';
          $output[] = '<td>' . ($data['last_transaction'] ?? 0) . '</td>';
          $output[] = '<td>' . $latest_txn . '</td>';
          $output[] = '<td>' . $transactions_qty . '</td>';
          $output[] = '<td><strong>' . $qty . '</strong></td>';
          $output[] = '</tr>';
        }
        $output[] = '</table>';
      }
    }

    $output[] = '<h3>' . $this->t('Final Stock Totals (All Variations Combined)') . '</h3>';
    $output[] = '<table border="1" cellpadding="5">';
    $output[] = '<tr><th>' . $this->t('Location') . '</th><th>' . $this->t('Total Stock') . '</th></tr>';
    foreach ($stock_by_location as $location_name => $qty) {
      $output[] = '<tr>';
      $output[] = '<td>' . $location_name . '</td>';
      $output[] = '<td><strong>' . $qty . '</strong></td>';
      $output[] = '</tr>';
    }
    $output[] = '</table>';

    return ['#markup' => implode("\n", $output)];
  }
}
