<?php

namespace Drupal\sogan_commerce_product\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\commerce_product\Entity\ProductInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\sogan_commerce_product\Service\ProductTableBuilder;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for AJAX product tables.
 */
class ProductTableController extends ControllerBase {

  /**
   * The product table builder.
   *
   * @var \Drupal\sogan_commerce_product\Service\ProductTableBuilder
   */
  protected $productTableBuilder;

  /**
   * Constructs a new ProductTableController object.
   *
   * @param \Drupal\sogan_commerce_product\Service\ProductTableBuilder $product_table_builder
   *   The product table builder.
   */
  public function __construct(ProductTableBuilder $product_table_builder) {
    $this->productTableBuilder = $product_table_builder;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('sogan_commerce_product.product_table_builder')
    );
  }

  /**
   * Returns a product table.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $commerce_product
   *   The product entity.
   * @param string $type
   *   The table type ('supplier' or 'variation').
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return array
   *   The render array.
   */
  public function getTable(ProductInterface $commerce_product, $type, Request $request) {
    $options = $request->query->all();
    
    // Sanitize boolean options passed as strings
    foreach ($options as $key => $value) {
      if ($value === 'true' || $value === '1') {
        $options[$key] = TRUE;
      } elseif ($value === 'false' || $value === '0') {
        $options[$key] = FALSE;
      }
    }

    if ($type === 'supplier') {
      $build = $this->productTableBuilder->buildSupplierTable($commerce_product, $options);
      $title = $this->t('Supplier Products for @product', ['@product' => $commerce_product->label()]);
    } elseif ($type === 'variation') {
      $build = $this->productTableBuilder->buildVariationTable($commerce_product, $options);
      $title = $this->t('Variations for @product', ['@product' => $commerce_product->label()]);
    } else {
      return ['#markup' => $this->t('Invalid table type.')];
    }

    // Wrap in a container for the modal
    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['product-table-modal-content']],
      'title' => [
        '#type' => 'html_tag',
        '#tag' => 'h2',
        '#value' => $title,
      ],
      'table' => $build,
    ];
  }

}
