<?php

namespace Drupal\sogan_commerce_product\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\sogan_commerce_product\Service\StockManager;
use Drush\Commands\DrushCommands;
use Drush\Attributes as CLI;

/**
 * Drush commands for managing product stock.
 */
final class StockCommands extends DrushCommands {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The stock manager service.
   *
   * @var \Drupal\sogan_commerce_product\Service\StockManager
   */
  protected $stockManager;

  /**
   * Constructs a StockCommands object.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, StockManager $stock_manager) {
    parent::__construct();
    $this->entityTypeManager = $entity_type_manager;
    $this->stockManager = $stock_manager;
  }

  /**
   * Check stock levels for a product.
   *
   * @param int $product_id
   *   The product ID.
   */
  #[CLI\Command(name: 'sogan:stock:check', aliases: ['ssc'])]
  #[CLI\Argument(name: 'product_id', description: 'Product ID')]
  #[CLI\Usage(name: 'sogan:stock:check 1', description: 'Check stock for product ID 1')]
  public function stockCheck($product_id) {
    $product_storage = $this->entityTypeManager->getStorage('commerce_product');
    $product = $product_storage->load($product_id);

    if (!$product) {
      $this->logger()->error("Product {$product_id} not found.");
      return;
    }

    $this->output()->writeln("Product: " . $product->getTitle() . " (ID: {$product_id})");
    $this->output()->writeln("");

    $location_storage = $this->entityTypeManager->getStorage('commerce_stock_location');
    $all_locations = $location_storage->loadMultiple();

    $this->output()->writeln("Stock Locations: " . count($all_locations));
    foreach ($all_locations as $location) {
      $this->output()->writeln("  - " . $location->getName() . " (ID: " . $location->id() . ")");
    }
    $this->output()->writeln("");

    $variations = $product->getVariations();
    $this->output()->writeln("Variations: " . count($variations));
    $this->output()->writeln("");

    foreach ($variations as $variation) {
      $this->output()->writeln("=== Variation ID: " . $variation->id() . " ===");
      $this->output()->writeln("Title: " . $variation->getTitle());
      
      foreach ($all_locations as $location) {
        $stock_level = $this->stockManager->getStockLevel($variation, $location);
        $this->output()->writeln("  " . $location->getName() . ": " . $stock_level);
        
        // Get transactions
        $transactions = $this->stockManager->getTransactions($variation, $location, 5);
        if (!empty($transactions)) {
          $this->output()->writeln("    Recent transactions:");
          foreach ($transactions as $txn) {
            $this->output()->writeln("      ID: {$txn->id}, Qty: {$txn->qty}, Type: {$txn->transaction_type_id}");
          }
        }
      }
      $this->output()->writeln("");
    }
  }

  /**
   * Set stock level for a variation.
   *
   * @param int $variation_id
   *   The variation ID.
   * @param int $location_id
   *   The location ID.
   * @param float $quantity
   *   The quantity.
   */
  #[CLI\Command(name: 'sogan:stock:set', aliases: ['sss'])]
  #[CLI\Argument(name: 'variation_id', description: 'Variation ID')]
  #[CLI\Argument(name: 'location_id', description: 'Location ID')]
  #[CLI\Argument(name: 'quantity', description: 'Quantity')]
  #[CLI\Usage(name: 'sogan:stock:set 2 1 15', description: 'Set stock for variation 2 at location 1 to 15')]
  public function stockSet($variation_id, $location_id, $quantity) {
    $variation_storage = $this->entityTypeManager->getStorage('commerce_product_variation');
    $location_storage = $this->entityTypeManager->getStorage('commerce_stock_location');
    
    $variation = $variation_storage->load($variation_id);
    $location = $location_storage->load($location_id);

    if (!$variation) {
      $this->logger()->error("Variation {$variation_id} not found.");
      return;
    }

    if (!$location) {
      $this->logger()->error("Location {$location_id} not found.");
      return;
    }

    $old_level = $this->stockManager->getStockLevel($variation, $location);
    $transaction_id = $this->stockManager->setStockLevel($variation, $location, (float)$quantity, 'Manual stock update via Drush');

    if ($transaction_id) {
      $this->logger()->success("Stock updated! Transaction ID: {$transaction_id}");
      $this->output()->writeln("Old level: {$old_level}");
      $this->output()->writeln("New level: {$quantity}");
    } else {
      $this->logger()->warning("No change needed. Current level is already {$quantity}");
    }
  }

  /**
   * Show stock transactions for a variation.
   *
   * @param int $variation_id
   *   The variation ID.
   * @param int|null $location_id
   *   Optional location ID to filter by.
   */
  #[CLI\Command(name: 'sogan:stock:transactions', aliases: ['sst'])]
  #[CLI\Argument(name: 'variation_id', description: 'Variation ID')]
  #[CLI\Option(name: 'location', description: 'Location ID to filter by')]
  #[CLI\Usage(name: 'sogan:stock:transactions 2', description: 'Show transactions for variation 2')]
  #[CLI\Usage(name: 'sogan:stock:transactions 2 --location=1', description: 'Show transactions for variation 2 at location 1')]
  public function stockTransactions($variation_id, $options = ['location' => NULL]) {
    $variation_storage = $this->entityTypeManager->getStorage('commerce_product_variation');
    $variation = $variation_storage->load($variation_id);

    if (!$variation) {
      $this->logger()->error("Variation {$variation_id} not found.");
      return;
    }

    $location = NULL;
    if ($options['location']) {
      $location_storage = $this->entityTypeManager->getStorage('commerce_stock_location');
      $location = $location_storage->load($options['location']);
      if (!$location) {
        $this->logger()->error("Location {$options['location']} not found.");
        return;
      }
    }

    $transactions = $this->stockManager->getTransactions($variation, $location, 50);

    if (empty($transactions)) {
      $this->output()->writeln("No transactions found.");
      return;
    }

    $this->output()->writeln("Transactions for Variation {$variation_id}:");
    $this->output()->writeln("");
    $this->output()->writeln(sprintf("%-6s %-12s %-8s %-15s %-10s", "ID", "Location", "Qty", "Type", "Time"));
    $this->output()->writeln(str_repeat("-", 60));

    foreach ($transactions as $txn) {
      $location_name = "";
      if ($txn->location_id) {
        $loc = $this->entityTypeManager->getStorage('commerce_stock_location')->load($txn->location_id);
        $location_name = $loc ? $loc->getName() : "Unknown";
      }
      
      $date = date('Y-m-d H:i', $txn->transaction_time);
      $this->output()->writeln(sprintf("%-6s %-12s %-8s %-15s %s", 
        $txn->id, 
        $location_name, 
        $txn->qty, 
        $txn->transaction_type_id,
        $date
      ));
    }
  }

}
