<?php

namespace Drupal\sogan_commerce_product\Commands;

use Consolidation\AnnotatedCommand\CommandData;
use Drush\Commands\DrushCommands;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Drush commands for the sogan_commerce_product module upgrades/repairs.
 */
class RepairMediaCommands extends DrushCommands
{
    use StringTranslationTrait;

    protected EntityTypeManagerInterface $entityTypeManager;

    public function __construct(EntityTypeManagerInterface $entity_type_manager)
    {
        parent::__construct();
        $this->entityTypeManager = $entity_type_manager;
    }

    /**
     * Repair media entities to ensure 'field_media_image' is populated.
     *
     * @command sogan_commerce_product:repair-media
     * @aliases scp-repair-media
     *
     * @param array $options
     *   Command options (unused currently).
     *
     * @option limit
     *   Limit processing to N media entities per run.
     *
     * @usage scp-repair-media
     *   Repair all media of bundle 'image'.
     */
    public function repairMedia(array $options = ['limit' => NULL])
    {
        $limit = $options['limit'] ?? NULL;
        $media_storage = $this->entityTypeManager->getStorage('media');
        $file_storage = $this->entityTypeManager->getStorage('file');
        $field_manager = \Drupal::service('entity_field.manager');

        $query = \Drupal::entityQuery('media')->condition('bundle', 'image')->accessCheck(FALSE);
        if ($limit) {
            $query->range(0, (int) $limit);
        }
        $mids = $query->execute();
        if (empty($mids)) {
            $this->output()->writeln($this->t('No media records to repair.'));
            return 0;
        }
        $count = 0;
        foreach (array_chunk($mids, 50) as $chunk) {
            $medias = $media_storage->loadMultiple($chunk);
            foreach ($medias as $media) {
                if (!$media->hasField('field_media_image') || !$media->get('field_media_image')->isEmpty()) {
                    continue;
                }

                $chosen_file_id = NULL;
                $field_defs = $field_manager->getFieldDefinitions('media', $media->bundle());
                foreach ($field_defs as $fname => $fdef) {
                    if ($fdef->getType() !== 'image') {
                        continue;
                    }
                    if ($media->get($fname)->isEmpty()) {
                        continue;
                    }
                    foreach ($media->get($fname)->getValue() as $value) {
                        if (empty($value['target_id'])) {
                            continue;
                        }
                        $fid = (int) $value['target_id'];
                        $file = $file_storage->load($fid);
                        if (!$file) {
                            continue;
                        }
                        if ($file->getFilename() === 'no-thumbnail.png') {
                            continue;
                        }
                        $chosen_file_id = $fid;
                        break 2;
                    }
                }

                if ($chosen_file_id) {
                    $media->set('field_media_image', [['target_id' => $chosen_file_id]]);
                    try {
                        $media->save();
                        $count++;
                        $this->output()->writeln($this->t('Updated media @id with file @fid', ['@id' => $media->id(), '@fid' => $chosen_file_id]));
                        continue;
                    } catch (\Exception $e) {
                        $this->output()->writeln($this->t('Failed to update media @id: @msg', ['@id' => $media->id(), '@msg' => $e->getMessage()]));
                    }
                }

                // Try remote_url basename search.
                if ($media->hasField('field_remote_url') && !$media->get('field_remote_url')->isEmpty()) {
                    $uri_val = $media->get('field_remote_url')->getValue()[0]['uri'] ?? '';
                    $base = basename(parse_url($uri_val, PHP_URL_PATH) ?? '');
                    if (!empty($base)) {
                        $file_query = \Drupal::entityQuery('file')->accessCheck(FALSE)->condition('uri', 'public://product_media/%' . $base . '%', 'LIKE');
                        $fids = $file_query->execute();
                        if (!empty($fids)) {
                            $fid = (int) reset($fids);
                            $media->set('field_media_image', [['target_id' => $fid]]);
                            try {
                                $media->save();
                                $count++;
                                $this->output()->writeln($this->t('Updated media @id via basename search with file @fid', ['@id' => $media->id(), '@fid' => $fid]));
                                continue;
                            } catch (\Exception $e) {
                                $this->output()->writeln($this->t('Failed to update media @id: @msg', ['@id' => $media->id(), '@msg' => $e->getMessage()]));
                            }
                        }
                    }
                }

                // Try sanitized name search.
                if ($media->hasField('name') && !$media->get('name')->isEmpty()) {
                    $safe_name = preg_replace('/[^A-Za-z0-9]+/', '_', $media->get('name')->value);
                    $file_query = \Drupal::entityQuery('file')->accessCheck(FALSE)->condition('uri', 'public://product_media/%' . $safe_name . '%', 'LIKE');
                    $fids = $file_query->execute();
                    if (!empty($fids)) {
                        $fid = (int) reset($fids);
                        $media->set('field_media_image', [['target_id' => $fid]]);
                        try {
                            $media->save();
                            $count++;
                            $this->output()->writeln($this->t('Updated media @id via name search with file @fid', ['@id' => $media->id(), '@fid' => $fid]));
                            continue;
                        } catch (\Exception $e) {
                            $this->output()->writeln($this->t('Failed to update media @id: @msg', ['@id' => $media->id(), '@msg' => $e->getMessage()]));
                        }
                    }
                }
            }
        }
        $this->output()->writeln($this->t('Total media updated: @count', ['@count' => $count]));
        return 0;
    }
}
