<?php

namespace Drupal\key_value_ifta_widget\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Provides an inline widget for key/value pairs.
 *
 * @FieldWidget(
 *   id = "key_value_inline_widget",
 *   label = @Translation("Key/value inline textfields"),
 *   field_types = {
 *     "key_value"
 *   }
 * )
 */
class InlineKeyValueWidget extends WidgetBase
{
    /**
     * {@inheritdoc}
     */
    public static function defaultSettings(): array
    {
        return [
            'key_placeholder' => '',
            'value_placeholder' => '',
        ] + parent::defaultSettings();
    }

    /**
     * {@inheritdoc}
     */
    public function settingsForm(array $form, FormStateInterface $form_state): array
    {
        $elements = parent::settingsForm($form, $form_state);

        $elements['key_placeholder'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Key placeholder'),
            '#default_value' => $this->getSetting('key_placeholder'),
        ];

        $elements['value_placeholder'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Value placeholder'),
            '#default_value' => $this->getSetting('value_placeholder'),
        ];

        return $elements;
    }

    /**
     * {@inheritdoc}
     */
    public function settingsSummary(): array
    {
        $summary = parent::settingsSummary();

        if ($key_placeholder = $this->getSetting('key_placeholder')) {
            $summary[] = $this->t('Key placeholder: @placeholder', ['@placeholder' => $key_placeholder]);
        }

        if ($value_placeholder = $this->getSetting('value_placeholder')) {
            $summary[] = $this->t('Value placeholder: @placeholder', ['@placeholder' => $value_placeholder]);
        }

        return $summary;
    }

    /**
     * {@inheritdoc}
     */
    public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array
    {
        $field_settings = $this->fieldDefinition->getSettings();
        $key_label = $field_settings['key_label'] ?? $this->t('Key');
        $value_label = $field_settings['value_label'] ?? $this->t('Value');
        $key_placeholder = $this->getSetting('key_placeholder') ?: ($field_settings['key_placeholder'] ?? '');
        $value_placeholder = $this->getSetting('value_placeholder') ?: ($field_settings['value_placeholder'] ?? '');

        $item = $items[$delta] ?? NULL;
        $key_default = $item && isset($item->key) ? $item->key : '';
        $value_default = $item && isset($item->value) ? $item->value : '';

        $build = [
            '#type' => 'container',
            '#attributes' => [
                'class' => ['key-value-ifta-widget'],
            ],
            '#attached' => [
                'library' => ['key_value_ifta_widget/inline-key-value'],
            ],
        ];

        $build['key'] = [
            '#type' => 'textfield',
            '#title' => $key_label,
            '#title_display' => 'before',
            '#default_value' => $key_default,
            '#maxlength' => 128,
            '#placeholder' => $key_placeholder,
            '#required' => FALSE,
            '#attributes' => [
                'class' => ['key-value-ifta-widget__input', 'key-value-ifta-widget__input--key'],
            ],
            '#wrapper_attributes' => [
                'class' => ['key-value-ifta-widget__item', 'key-value-ifta-widget__item--key', 'ifta-label'],
            ],
        ];

        $build['value'] = [
            '#type' => 'textfield',
            '#title' => $value_label,
            '#title_display' => 'before',
            '#default_value' => $value_default,
            '#maxlength' => 512,
            '#placeholder' => $value_placeholder,
            '#required' => FALSE,
            '#attributes' => [
                'class' => ['key-value-ifta-widget__input', 'key-value-ifta-widget__input--value'],
            ],
            '#wrapper_attributes' => [
                'class' => ['key-value-ifta-widget__item', 'key-value-ifta-widget__item--value', 'ifta-label'],
            ],
        ];

        if ($value_default === '' && $key_default === '') {
            $build['key']['#states'] = [
                'required' => [
                    ':input[name="' . $this->fieldDefinition->getName() . '[' . $delta . '][value]"]' => ['empty' => FALSE],
                ],
            ];
        }

        $build['#element_validate'][] = [static::class, 'validateKey'];

        return $build;
    }

    /**
     * Require key when value present.
     */
    public static function validateKey(array $element, FormStateInterface $form_state): void
    {
        $value = $element['value']['#value'] ?? '';
        $key = $element['key']['#value'] ?? '';
        if ($value !== '' && $key === '') {
            $message = new TranslatableMarkup('@name field is required when a value is provided.', ['@name' => $element['key']['#title']]);
            $form_state->setError($element['key'], $message);
        }
    }
}
