<?php

namespace Drupal\feeds_performance;

use Drupal\feeds\Result\ParserResultInterface;
use Drupal\feeds\StateInterface;
use Drupal\feeds\FeedInterface;
use Drupal\feeds\Feeds\Item\ItemInterface;
use IteratorAggregate;
use Traversable;

/**
 * Wraps a ParserResult to add performance checks.
 */
class PerformanceParserResult implements ParserResultInterface
{

    /**
     * The original parser result.
     *
     * @var \Drupal\feeds\Result\ParserResultInterface
     */
    protected $innerResult;

    /**
     * The hasher service.
     *
     * @var \Drupal\feeds_performance\FeedsPerformanceHasher
     */
    protected $hasher;

    /**
     * The feed being imported.
     *
     * @var \Drupal\feeds\FeedInterface
     */
    protected $feed;

    /**
     * The feeds state.
     *
     * @var \Drupal\feeds\StateInterface
     */
    protected $state;

    /**
     * The iterator for filtering items.
     *
     * @var \Generator
     */
    protected $iterator;

    /**
     * Constructs a PerformanceParserResult object.
     *
     * @param \Drupal\feeds\Result\ParserResultInterface $inner_result
     *   The original parser result.
     * @param \Drupal\feeds_performance\FeedsPerformanceHasher $hasher
     *   The hasher service.
     * @param \Drupal\feeds\FeedInterface $feed
     *   The feed being imported.
     * @param \Drupal\feeds\StateInterface $state
     *   The feeds state.
     */
    public function __construct(ParserResultInterface $inner_result, FeedsPerformanceHasher $hasher, FeedInterface $feed, StateInterface $state)
    {
        $this->innerResult = $inner_result;
        $this->hasher = $hasher;
        $this->feed = $feed;
        $this->state = $state;
        $this->iterator = $this->createIterator();
    }

    /**
     * Creates the filtering iterator.
     *
     * @return \Generator
     */
    protected function createIterator()
    {
        foreach ($this->innerResult as $item) {
            if ($item instanceof SkippedItem) {
                $this->state->skipped++;
                continue;
            }

            if (!$this->hasher->shouldImportItem($this->feed, $item)) {
                $this->state->skipped++;
                continue;
            }

            yield $item;
        }
    }

    /**
     * {@inheritdoc}
     */
    #[\ReturnTypeWillChange]
    public function current()
    {
        return $this->iterator->current();
    }

    /**
     * {@inheritdoc}
     */
    #[\ReturnTypeWillChange]
    public function next()
    {
        $this->iterator->next();
    }

    /**
     * {@inheritdoc}
     */
    #[\ReturnTypeWillChange]
    public function key()
    {
        return $this->iterator->key();
    }

    /**
     * {@inheritdoc}
     */
    #[\ReturnTypeWillChange]
    public function valid()
    {
        return $this->iterator->valid();
    }

    /**
     * {@inheritdoc}
     */
    #[\ReturnTypeWillChange]
    public function rewind()
    {
        $this->iterator = $this->createIterator();
    }

    /**
     * {@inheritdoc}
     */
    public function addItem(ItemInterface $item)
    {
        $this->innerResult->addItem($item);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function addItems(array $items)
    {
        $this->innerResult->addItems($items);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    #[\ReturnTypeWillChange]
    public function count()
    {
        return $this->innerResult->count();
    }

    /**
     * {@inheritdoc}
     */
    #[\ReturnTypeWillChange]
    public function offsetExists($offset)
    {
        return $this->innerResult->offsetExists($offset);
    }

    /**
     * {@inheritdoc}
     */
    #[\ReturnTypeWillChange]
    public function offsetGet($offset)
    {
        return $this->innerResult->offsetGet($offset);
    }

    /**
     * {@inheritdoc}
     */
    #[\ReturnTypeWillChange]
    public function offsetSet($offset, $value)
    {
        $this->innerResult->offsetSet($offset, $value);
    }

    /**
     * {@inheritdoc}
     */
    #[\ReturnTypeWillChange]
    public function offsetUnset($offset)
    {
        $this->innerResult->offsetUnset($offset);
    }
}
