<?php

namespace Drupal\feeds_performance\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\feeds_performance\FeedsPerformanceHasher;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Feeds Performance settings from the Synchronization page.
 */
class FeedsPerformanceSynchronizationForm extends FormBase
{

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected EntityTypeManagerInterface $entityTypeManager;

    /**
     * The feeds performance hasher service.
     *
     * @var \Drupal\feeds_performance\FeedsPerformanceHasher
     */
    protected FeedsPerformanceHasher $hasher;

    /**
     * Constructs a FeedsPerformanceSynchronizationForm object.
     */
    public function __construct(EntityTypeManagerInterface $entity_type_manager, FeedsPerformanceHasher $hasher)
    {
        $this->entityTypeManager = $entity_type_manager;
        $this->hasher = $hasher;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container)
    {
        return new static(
            $container->get('entity_type.manager'),
            $container->get('feeds_performance.hasher')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function getFormId()
    {
        return 'feeds_performance_synchronization_settings';
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(array $form, FormStateInterface $form_state)
    {
        $feed_types = $this->entityTypeManager->getStorage('feeds_feed_type')->loadMultiple();

        if (empty($feed_types)) {
            $form['empty'] = [
                '#markup' => $this->t('No feed types found. Please create a feed type first.'),
            ];
            return $form;
        }

        $form['description'] = [
            '#markup' => '<p>' . $this->t('Configure performance settings for each feed type. Item Context is configured per feed type in the feed type settings.') . '</p>',
        ];

        $form['feed_types'] = [
            '#type' => 'table',
            '#header' => [
                $this->t('Feed Type'),
                $this->t('Import Period'),
                $this->t('Check File Update'),
                $this->t('Check Item Update'),
                $this->t('Force Update'),
                $this->t('Debug Logs'),
                $this->t('Actions'),
            ],
            '#empty' => $this->t('No feed types available.'),
        ];

        // Build import period options (same as FeedTypeForm + custom 1-10 minute intervals)
        $import_period_options = $this->getImportPeriodOptions();

        /** @var \Drupal\feeds\FeedTypeInterface $feed_type */
        foreach ($feed_types as $feed_type_id => $feed_type) {
            $form['feed_types'][$feed_type_id]['label'] = [
                '#markup' => '<strong>' . $feed_type->label() . '</strong>',
            ];

            $form['feed_types'][$feed_type_id]['import_period'] = [
                '#type' => 'select',
                '#title' => $this->t('Import Period'),
                '#title_display' => 'invisible',
                '#options' => $import_period_options,
                '#default_value' => $feed_type->getImportPeriod(),
            ];

            $form['feed_types'][$feed_type_id]['check_file_xml_update'] = [
                '#type' => 'checkbox',
                '#title' => $this->t('Enabled'),
                '#title_display' => 'invisible',
                '#default_value' => $feed_type->getThirdPartySetting('feeds_performance', 'check_file_xml_update', TRUE),
            ];

            $form['feed_types'][$feed_type_id]['check_item_xml_update'] = [
                '#type' => 'checkbox',
                '#title' => $this->t('Enabled'),
                '#title_display' => 'invisible',
                '#default_value' => $feed_type->getThirdPartySetting('feeds_performance', 'check_item_xml_update', FALSE),
            ];

            $form['feed_types'][$feed_type_id]['force_update'] = [
                '#type' => 'checkbox',
                '#title' => $this->t('Enabled'),
                '#title_display' => 'invisible',
                '#default_value' => $feed_type->getProcessor()->getConfiguration('skip_hash_check') ?? FALSE,
            ];

            $form['feed_types'][$feed_type_id]['debug_logs'] = [
                '#type' => 'checkbox',
                '#title' => $this->t('Enabled'),
                '#title_display' => 'invisible',
                '#default_value' => $feed_type->getThirdPartySetting('feeds_performance', 'debug_logs', FALSE),
            ];

            $form['feed_types'][$feed_type_id]['clear_hashes'] = [
                '#type' => 'submit',
                '#value' => $this->t('Clear Hashes'),
                '#name' => 'clear_hashes_' . $feed_type_id,
                '#submit' => ['::clearHashesForFeedTypeSubmit'],
                '#limit_validation_errors' => [],
                '#feed_type_id' => $feed_type_id,
                '#attributes' => [
                    'class' => ['button--small'],
                ],
            ];
        }

        $form['actions'] = [
            '#type' => 'actions',
        ];

        $form['actions']['submit'] = [
            '#type' => 'submit',
            '#value' => $this->t('Save configuration'),
            '#button_type' => 'primary',
        ];

        $form['actions']['clear_all_hashes'] = [
            '#type' => 'submit',
            '#value' => $this->t('Clear All Hashes'),
            '#submit' => ['::clearHashesSubmit'],
            '#limit_validation_errors' => [],
        ];

        return $form;
    }

    /**
     * {@inheritdoc}
     */
    public function submitForm(array &$form, FormStateInterface $form_state)
    {
        $feed_type_values = $form_state->getValue('feed_types');
        $feed_type_storage = $this->entityTypeManager->getStorage('feeds_feed_type');

        foreach ($feed_type_values as $feed_type_id => $values) {
            /** @var \Drupal\feeds\FeedTypeInterface $feed_type */
            $feed_type = $feed_type_storage->load($feed_type_id);
            if ($feed_type) {
                // Update import period (native Feeds setting)
                $feed_type->setImportPeriod((int) $values['import_period']);

                $feed_type->setThirdPartySetting('feeds_performance', 'check_file_xml_update', (bool) $values['check_file_xml_update']);
                $feed_type->setThirdPartySetting('feeds_performance', 'check_item_xml_update', (bool) $values['check_item_xml_update']);
                $feed_type->setThirdPartySetting('feeds_performance', 'debug_logs', (bool) $values['debug_logs']);

                // Update the processor's skip_hash_check setting (native Feeds "Force update")
                $processor = $feed_type->getProcessor();
                $processor_config = $processor->getConfiguration();
                $processor_config['skip_hash_check'] = (bool) $values['force_update'];
                $processor->setConfiguration($processor_config);

                $feed_type->save();
            }
        }

        $this->messenger()->addStatus($this->t('The configuration has been saved.'));
    }

    /**
     * Submit handler for clearing hashes for a specific feed type.
     */
    public function clearHashesForFeedTypeSubmit(array &$form, FormStateInterface $form_state)
    {
        $triggering_element = $form_state->getTriggeringElement();
        $feed_type_id = $triggering_element['#feed_type_id'] ?? NULL;

        if ($feed_type_id) {
            $count = $this->hasher->clearHashesForFeedType($feed_type_id);
            $feed_type = $this->entityTypeManager->getStorage('feeds_feed_type')->load($feed_type_id);
            $label = $feed_type ? $feed_type->label() : $feed_type_id;

            $this->messenger()->addStatus($this->t('Cleared performance hashes for @count feeds of type "@type".', [
                '@count' => $count,
                '@type' => $label,
            ]));
        }
    }

    /**
     * Submit handler for clearing all hashes.
     */
    public function clearHashesSubmit(array &$form, FormStateInterface $form_state)
    {
        $deleted = $this->hasher->clearAllHashes();

        $this->messenger()->addStatus($this->t('Cleared all performance hashes: @file_hashes file hashes and @item_hashes item hashes deleted.', [
            '@file_hashes' => $deleted['file_hashes'],
            '@item_hashes' => $deleted['item_hashes'],
        ]));
    }

    /**
     * Returns the available import period options.
     *
     * @return array
     *   An array of import period options.
     */
    protected function getImportPeriodOptions(): array
    {
        $dateFormatter = \Drupal::service('date.formatter');

        // Standard Feeds import periods
        $times = [
            900,
            1800,
            3600,
            10800,
            21600,
            43200,
            86400,
            259200,
            604800,
            2419200,
        ];

        $period = array_map(function ($time) use ($dateFormatter) {
            return $this->t('Every @p', ['@p' => $dateFormatter->formatInterval($time)]);
        }, array_combine($times, $times));

        // Add custom 1-10 minute intervals
        $custom = [];
        for ($minutes = 1; $minutes <= 10; $minutes++) {
            $seconds = $minutes * 60;
            if (!isset($period[$seconds])) {
                $custom[$seconds] = $minutes === 1
                    ? $this->t('Every minute')
                    : $this->t('Every @count minutes', ['@count' => $minutes]);
            }
        }

        // Build final options array with Off and As often as possible first
        $options = [
            \Drupal\feeds\FeedTypeInterface::SCHEDULE_NEVER => $this->t('Off'),
            \Drupal\feeds\FeedTypeInterface::SCHEDULE_CONTINUOUSLY => $this->t('As often as possible'),
        ] + $custom + $period;

        return $options;
    }
}
