<?php

namespace Drupal\feeds_performance\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Feeds Performance settings.
 */
class FeedsPerformanceSettingsForm extends ConfigFormBase
{

    /**
     * {@inheritdoc}
     */
    public function getFormId()
    {
        return 'feeds_performance_settings';
    }

    /**
     * {@inheritdoc}
     */
    protected function getEditableConfigNames()
    {
        return ['feeds_performance.settings'];
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(array $form, FormStateInterface $form_state)
    {
        $config = $this->config('feeds_performance.settings');

        $form['garbage_collection'] = [
            '#type' => 'details',
            '#title' => $this->t('Garbage Collection'),
            '#open' => TRUE,
        ];

        $form['garbage_collection']['enable_gc'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Enable daily garbage collection with cron'),
            '#description' => $this->t('Automatically clean up old hash records during cron runs.'),
            '#default_value' => $config->get('enable_gc') ?? FALSE,
        ];

        $form['garbage_collection']['gc_hour'] = [
            '#type' => 'select',
            '#title' => $this->t('Trigger hour'),
            '#description' => $this->t('Hour of the day to run garbage collection (24-hour format).'),
            '#options' => array_combine(range(0, 23), array_map(function ($h) {
                return sprintf('%02d:00', $h);
            }, range(0, 23))),
            '#default_value' => $config->get('gc_hour') ?? 3,
            '#states' => [
                'visible' => [
                    ':input[name="enable_gc"]' => ['checked' => TRUE],
                ],
            ],
        ];

        $form['garbage_collection']['gc_minute'] = [
            '#type' => 'select',
            '#title' => $this->t('Trigger minute'),
            '#description' => $this->t('Minute of the hour to run garbage collection.'),
            '#options' => array_combine(range(0, 59), array_map(function ($m) {
                return sprintf('%02d', $m);
            }, range(0, 59))),
            '#default_value' => $config->get('gc_minute') ?? 0,
            '#states' => [
                'visible' => [
                    ':input[name="enable_gc"]' => ['checked' => TRUE],
                ],
            ],
        ];

        $form['garbage_collection']['gc_days_threshold'] = [
            '#type' => 'number',
            '#title' => $this->t('Days threshold'),
            '#description' => $this->t('Delete hash records older than this many days. Default: 30 days.'),
            '#default_value' => $config->get('gc_days_threshold') ?? 30,
            '#min' => 1,
            '#max' => 365,
            '#states' => [
                'visible' => [
                    ':input[name="enable_gc"]' => ['checked' => TRUE],
                ],
            ],
        ];

        $form['actions_wrapper'] = [
            '#type' => 'details',
            '#title' => $this->t('Manual Actions'),
            '#open' => TRUE,
        ];

        $form['actions_wrapper']['clear_all_hashes'] = [
            '#type' => 'submit',
            '#value' => $this->t('Clear All Hashes'),
            '#submit' => ['::clearHashesSubmit'],
            '#limit_validation_errors' => [],
            '#description' => $this->t('Manually clear all performance hashes from the database. This will force a full re-import for all feeds on their next run.'),
        ];

        return parent::buildForm($form, $form_state);
    }

    /**
     * {@inheritdoc}
     */
    public function submitForm(array &$form, FormStateInterface $form_state)
    {
        $this->config('feeds_performance.settings')
            ->set('enable_gc', $form_state->getValue('enable_gc'))
            ->set('gc_hour', $form_state->getValue('gc_hour'))
            ->set('gc_minute', $form_state->getValue('gc_minute'))
            ->set('gc_days_threshold', $form_state->getValue('gc_days_threshold'))
            ->save();

        parent::submitForm($form, $form_state);
    }

    /**
     * Submit handler for clearing all hashes.
     */
    public function clearHashesSubmit(array &$form, FormStateInterface $form_state)
    {
        /** @var \Drupal\feeds_performance\FeedsPerformanceHasher $hasher */
        $hasher = \Drupal::service('feeds_performance.hasher');
        $deleted = $hasher->clearAllHashes();

        $this->messenger()->addStatus($this->t('Cleared all performance hashes: @file_hashes file hashes and @item_hashes item hashes deleted.', [
            '@file_hashes' => $deleted['file_hashes'],
            '@item_hashes' => $deleted['item_hashes'],
        ]));
    }
}
