<?php

namespace Drupal\feeds_performance;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Database\Connection;
use Drupal\feeds\FeedInterface;
use DOMDocument;
use DOMXPath;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service to calculate and check hashes.
 */
class FeedsPerformanceHasher
{

    /**
     * The database connection.
     *
     * @var \Drupal\Core\Database\Connection
     */
    protected $database;

    /**
     * The logger service.
     *
     * @var \Drupal\Core\Logger\LoggerChannelInterface
     */
    protected $logger;

    /**
     * The time service.
     *
     * @var \Drupal\Component\Datetime\TimeInterface
     */
    protected $time;

    /**
     * Constructs a FeedsPerformanceHasher object.
     */
    public function __construct(Connection $database, LoggerChannelFactoryInterface $logger_factory, TimeInterface $time)
    {
        $this->database = $database;
        $this->logger = $logger_factory->get('feeds_performance');
        $this->time = $time;
    }

    /**
     * Checks if the file hash has changed.
     *
     * @param int $feed_id
     *   The feed ID.
     * @param string $hash
     *   The current file hash.
     *
     * @return bool
     *   TRUE if the hash is different (changed), FALSE otherwise.
     */
    public function hasFileChanged($feed_id, $hash)
    {
        $stored_hash = $this->database->select('feeds_performance_file_hash', 'f')
            ->fields('f', ['hash'])
            ->condition('feed_id', $feed_id)
            ->execute()
            ->fetchField();

        return $stored_hash !== $hash;
    }

    /**
     * Updates the file hash.
     *
     * @param int $feed_id
     *   The feed ID.
     * @param string $hash
     *   The new file hash.
     */
    public function updateFileHash($feed_id, $hash)
    {
        $this->logger->info('updateFileHash called. feed_id: @feed_id, hash: @hash', [
            '@feed_id' => $feed_id,
            '@hash' => $hash,
        ]);

        $this->database->merge('feeds_performance_file_hash')
            ->key('feed_id', $feed_id)
            ->fields([
                'hash' => $hash,
                'timestamp' => $this->time->getRequestTime(),
            ])
            ->execute();
    }

    /**
     * Checks if an item hash has changed.
     *
     * @param int $feed_id
     *   The feed ID.
     * @param string $item_key_hash
     *   The hash of the item's unique key.
     * @param string $item_xml_hash
     *   The hash of the item's XML content.
     *
     * @return bool
     *   TRUE if the hash is different (changed/new), FALSE otherwise.
     */
    public function hasItemChanged($feed_id, $item_key_hash, $item_xml_hash)
    {
        $stored_xml_hash = $this->database->select('feeds_performance_item_hash', 'i')
            ->fields('i', ['item_xml_hash'])
            ->condition('feed_id', $feed_id)
            ->condition('item_key_hash', $item_key_hash)
            ->execute()
            ->fetchField();

        // If no record exists, it's new/changed.
        if (!$stored_xml_hash) {
            return TRUE;
        }

        // If stored hash differs from current hash, it's changed.
        return $stored_xml_hash !== $item_xml_hash;
    }

    /**
     * Updates the item hash.
     *
     * @param int $feed_id
     *   The feed ID.
     * @param string $item_key_hash
     *   The hash of the item's unique key.
     * @param string $item_xml_hash
     *   The hash of the item's XML content.
     */
    public function updateItemHash($feed_id, $item_key_hash, $item_xml_hash)
    {
        $this->database->merge('feeds_performance_item_hash')
            ->key('feed_id', $feed_id)
            ->key('item_key_hash', $item_key_hash)
            ->fields([
                'item_xml_hash' => $item_xml_hash,
                'timestamp' => $this->time->getRequestTime(),
            ])
            ->execute();
    }

    /**
     * Deletes all item hashes for a feed.
     * 
     * @param int $feed_id
     *   The feed ID.
     */
    public function deleteItemHashes($feed_id)
    {
        $this->database->delete('feeds_performance_item_hash')
            ->condition('feed_id', $feed_id)
            ->execute();
    }
    /**
     * Checks if an item should be imported based on performance settings.
     *
     * @param \Drupal\feeds\FeedInterface $feed
     *   The feed object.
     * @param string|\Drupal\feeds\Feeds\Item\ItemInterface $item
     *   The XML content of the item or the Item object.
     *
     * @return bool
     *   TRUE if the item should be imported, FALSE if it should be skipped.
     */
    public function shouldImportItem(FeedInterface $feed, $item)
    {
        $check_item_update = $feed->getType()->getThirdPartySetting('feeds_performance', 'check_item_xml_update', FALSE);
        $skip_hash_check = $feed->getType()->getProcessor()->getConfiguration('skip_hash_check') ?? FALSE;

        // If skip_hash_check (Force update) is enabled, always import
        if ($skip_hash_check) {
            return TRUE;
        }

        // If check is disabled, always import
        if (empty($check_item_update)) {
            return TRUE;
        }

        $debug_logs = $feed->getType()->getThirdPartySetting('feeds_performance', 'debug_logs', FALSE);

        // Handle XML string (StreamingXmlParser)
        if (is_string($item)) {
            $item_xml = $item;
            $item_xml_hash = hash('sha256', $item_xml);
            $item_key_hash = '';
            $item_key_value = '';
            $item_context = $feed->getType()->getThirdPartySetting('feeds_performance', 'item_context', '');

            // If item context is configured, extract the key
            if (!empty($item_context)) {
                $dom = new DOMDocument();
                $previous_error_handling = libxml_use_internal_errors(TRUE);
                if (@$dom->loadXML($item_xml)) {
                    $xpath = new DOMXPath($dom);
                    // Handle relative xpath if needed
                    $query = $item_context;
                    if (strpos($query, '/') !== 0) {
                        $query = './' . $query;
                    }

                    // Query relative to the root element
                    $nodes = $xpath->query($query, $dom->documentElement);
                    if ($nodes && $nodes->length > 0) {
                        $item_key_value = $nodes->item(0)->nodeValue;
                        $item_key_hash = hash('sha256', $item_key_value);
                    }
                } else {
                    $errors = libxml_get_errors();
                    if ($errors) {
                        $this->logger->warning('Failed to parse item XML for key extraction: @errors', [
                            '@errors' => json_encode($errors)
                        ]);
                    }
                }
                libxml_use_internal_errors($previous_error_handling);
                libxml_clear_errors();
            }

            // Fallback to xml hash if no key found or not configured
            if (empty($item_key_hash)) {
                $item_key_hash = $item_xml_hash;
                $item_key_value = 'Hash: ' . $item_xml_hash;
            }

            if (!$this->hasItemChanged($feed->id(), $item_key_hash, $item_xml_hash)) {
                if ($debug_logs) {
                    $this->logger->info('Skipping item @key (Context: @context) for feed @feed_id because it has not changed.<br><pre>@xml</pre>', [
                        '@key' => $item_key_value,
                        '@context' => $item_context,
                        '@feed_id' => $feed->id(),
                        '@xml' => $item_xml
                    ]);
                }
                return FALSE;
            }

            $this->updateItemHash($feed->id(), $item_key_hash, $item_xml_hash);
            return TRUE;
        }

        // Handle ItemInterface (Generic)
        if ($item instanceof \Drupal\feeds\Feeds\Item\ItemInterface) {
            // Generic hashing strategy: hash the serialized item data
            // We use toArray() to get the raw values
            $item_data = $item->toArray();
            // Sort keys to ensure consistent order
            ksort($item_data);
            $item_xml_hash = hash('sha256', serialize($item_data));

            // For generic items, we don't have a specific "key" unless we configure one.
            // But Feeds items don't have a standard "ID" field until mapped.
            // So we use the hash as the key for now.
            // Ideally, we would use a configured source as the key, similar to 'item_context'.
            // But 'item_context' is XPath specific.
            // For now, we just check if the content changed.
            $item_key_hash = $item_xml_hash;
            $item_key_value = 'Generic Hash: ' . $item_xml_hash;

            if (!$this->hasItemChanged($feed->id(), $item_key_hash, $item_xml_hash)) {
                if ($debug_logs) {
                    $this->logger->info('Skipping generic item for feed @feed_id because it has not changed.', [
                        '@feed_id' => $feed->id()
                    ]);
                }
                return FALSE;
            }

            $this->updateItemHash($feed->id(), $item_key_hash, $item_xml_hash);
            return TRUE;
        }

        return TRUE;
    }

    /**
     * Deletes all hashes for a specific feed.
     *
     * @param int $feed_id
     *   The feed ID.
     */
    public function deleteAllHashesForFeed($feed_id)
    {
        // Delete file hash
        $this->database->delete('feeds_performance_file_hash')
            ->condition('feed_id', $feed_id)
            ->execute();

        // Delete item hashes
        $this->deleteItemHashes($feed_id);
    }

    /**
     * Clears all hashes for feeds of a specific feed type.
     *
     * @param string $feed_type_id
     *   The feed type ID.
     *
     * @return int
     *   The number of feeds cleared.
     */
    public function clearHashesForFeedType($feed_type_id)
    {
        // Find all feeds of this type
        $query = $this->database->select('feeds_feed', 'f');
        $query->fields('f', ['fid']);
        $query->condition('type', $feed_type_id);
        $fids = $query->execute()->fetchCol();

        if (empty($fids)) {
            return 0;
        }

        // Delete file hashes
        $this->database->delete('feeds_performance_file_hash')
            ->condition('feed_id', $fids, 'IN')
            ->execute();

        // Delete item hashes
        $this->database->delete('feeds_performance_item_hash')
            ->condition('feed_id', $fids, 'IN')
            ->execute();

        return count($fids);
    }

    /**
     * Cleans up hash records older than the specified threshold.
     *
     * @param int $days_threshold
     *   Delete records older than this many days.
     *
     * @return array
     *   Array with 'file_hashes' and 'item_hashes' counts deleted.
     */
    public function cleanupOldHashes($days_threshold = 30)
    {
        $threshold_timestamp = $this->time->getRequestTime() - ($days_threshold * 86400);

        // Delete old file hashes
        $file_deleted = $this->database->delete('feeds_performance_file_hash')
            ->condition('timestamp', $threshold_timestamp, '<')
            ->execute();

        // Delete old item hashes
        $item_deleted = $this->database->delete('feeds_performance_item_hash')
            ->condition('timestamp', $threshold_timestamp, '<')
            ->execute();

        return [
            'file_hashes' => $file_deleted,
            'item_hashes' => $item_deleted,
        ];
    }
    /**
     * Clears all hashes from the database.
     *
     * @return array
     *   Array with 'file_hashes' and 'item_hashes' counts deleted.
     */
    public function clearAllHashes()
    {
        // Delete all file hashes
        $file_deleted = $this->database->delete('feeds_performance_file_hash')
            ->execute();

        // Delete all item hashes
        $item_deleted = $this->database->delete('feeds_performance_item_hash')
            ->execute();

        return [
            'file_hashes' => $file_deleted,
            'item_hashes' => $item_deleted,
        ];
    }
}
