<?php

namespace Drupal\feeds_performance\EventSubscriber;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\feeds\Event\FeedsEvents;
use Drupal\feeds\Event\ParseEvent;
use Drupal\feeds\Exception\EmptyFeedException;
use Drupal\feeds\StateInterface as FeedsStateInterface;
use Drupal\feeds_performance\FeedsPerformanceHasher;
use Drupal\feeds_performance\PerformanceParserResult;
use Drupal\Core\State\StateInterface;
use Drupal\feeds\Event\ImportFinishedEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Subscribes to Feeds events for performance checks.
 */
class FeedsPerformanceSubscriber implements EventSubscriberInterface
{

    /**
     * The hasher service.
     *
     * @var \Drupal\feeds_performance\FeedsPerformanceHasher
     */
    protected $hasher;

    /**
     * The messenger service.
     *
     * @var \Drupal\Core\Messenger\MessengerInterface
     */
    protected $messenger;

    /**
     * The state service.
     *
     * @var \Drupal\Core\State\StateInterface
     */
    protected $state;

    /**
     * The logger.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * Constructs a FeedsPerformanceSubscriber object.
     */
    public function __construct(FeedsPerformanceHasher $hasher, MessengerInterface $messenger, StateInterface $state, LoggerChannelFactoryInterface $logger_factory)
    {
        $this->hasher = $hasher;
        $this->messenger = $messenger;
        $this->state = $state;
        $this->logger = $logger_factory->get('feeds_performance');
    }

    /**
     * {@inheritdoc}
     */
    public static function getSubscribedEvents()
    {
        // High priority to run before the parser.
        return [
            FeedsEvents::PARSE => ['onParse', 1000],
            FeedsEvents::IMPORT_FINISHED => ['onImportFinished', 100],
        ];
    }

    /**
     * Checks file hash on parse initialization.
     *
     * @param \Drupal\feeds\Event\ParseEvent $event
     *   The parse event.
     */
    public function onParse(ParseEvent $event)
    {
        $feed = $event->getFeed();
        $fetcher_result = $event->getFetcherResult();
        $raw = $fetcher_result->getRaw();

        $file_path = $fetcher_result->getFilePath();

        if (!is_string($file_path)) {
            return;
        }

        // Check if performance check is enabled.
        $check_file_update = $feed->getType()->getThirdPartySetting('feeds_performance', 'check_file_xml_update', TRUE);
        $skip_hash_check = $feed->getType()->getProcessor()->getConfiguration('skip_hash_check') ?? FALSE;
        $debug_logs = $feed->getType()->getThirdPartySetting('feeds_performance', 'debug_logs', FALSE);

        // If skip_hash_check (Force update) is enabled, skip all hash checking
        if ($skip_hash_check) {
            if ($debug_logs) {
                $this->logger->info('Force update enabled for feed @feed_id. Skipping all hash checks.', ['@feed_id' => $feed->id()]);
            }
            return;
        }

        if (empty($check_file_update)) {
            return;
        }

        // Calculate hash of the file
        $hash = hash_file('sha256', $file_path);

        // Check if file has changed using the hasher service (Database)
        if (!$this->hasher->hasFileChanged($feed->id(), $hash)) {
            // File has not changed, skip import
            if ($debug_logs) {
                $this->logger->info('Skipping import for feed @feed_id because file has not changed.', ['@feed_id' => $feed->id()]);
            }
            throw new EmptyFeedException();
        }

        // Store hash in state to be saved on import finished
        // We use a temporary state key to avoid updating the actual hash if the import fails
        $temp_state_key = 'feeds_performance.temp_file_hash.' . $feed->id();
        $this->state->set($temp_state_key, $hash);

        if ($debug_logs) {
            $this->logger->info('File hash changed for feed @feed_id. Proceeding with import.', ['@feed_id' => $feed->id()]);
        }

        // Wrap the parser result to handle item skipping and progress reporting
        $parser_result = $event->getParserResult();
        if ($parser_result) {
            $feed_state = $feed->getState(FeedsStateInterface::PARSE);
            $performance_result = new PerformanceParserResult($parser_result, $this->hasher, $feed, $feed_state);
            $event->setParserResult($performance_result);
        }
    }

    /**
     * Updates file hash when import is finished.
     *
     * @param \Drupal\feeds\Event\ImportFinishedEvent $event
     *   The import finished event.
     */
    public function onImportFinished(ImportFinishedEvent $event)
    {
        $feed = $event->getFeed();
        $feed_id = $feed->id();
        // Use the same key as in onParse
        $key = 'feeds_performance.temp_file_hash.' . $feed_id;
        $hash = $this->state->get($key);

        if ($hash) {
            $this->hasher->updateFileHash($feed_id, $hash);
            $this->state->delete($key);
        }
    }
}
