<?php

namespace Drupal\commerce_shipping_label;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_shipping\Entity\ShipmentInterface;

/**
 * Interface for shipping label generator service.
 */
interface ShippingLabelGeneratorInterface
{

    /**
     * Generates a PDF label for an order.
     *
     * @param \Drupal\commerce_order\Entity\OrderInterface $order
     *   The order entity.
     * @param string|null $template_id
     *   The template ID to use, or NULL for default.
     *
     * @return string
     *   The PDF content as a string.
     */
    public function generateFromOrder(OrderInterface $order, ?string $template_id = NULL): string;

    /**
     * Generates a PDF label for a shipment.
     *
     * @param \Drupal\commerce_shipping\Entity\ShipmentInterface $shipment
     *   The shipment entity.
     * @param string|null $template_id
     *   The template ID to use, or NULL for default.
     *
     * @return string
     *   The PDF content as a string.
     */
    public function generateFromShipment(ShipmentInterface $shipment, ?string $template_id = NULL): string;

    /**
     * Generates a single PDF with multiple labels (bulk mode).
     *
     * @param array $entities
     *   Array of order or shipment entities.
     * @param string|null $template_id
     *   The template ID to use, or NULL for default.
     *
     * @return string
     *   The PDF content as a string.
     */
    public function generateBulk(array $entities, ?string $template_id = NULL): string;

    /**
     * Builds render array for a label.
     *
     * @param \Drupal\commerce_order\Entity\OrderInterface|\Drupal\commerce_shipping\Entity\ShipmentInterface $entity
     *   The order or shipment entity.
     * @param string|null $template_id
     *   The template ID to use, or NULL for default.
     *
     * @return array
     *   A render array for the label.
     */
    public function buildLabelRenderArray($entity, ?string $template_id = NULL): array;

    /**
     * Gets available templates.
     *
     * @return array
     *   Array of template definitions keyed by ID.
     */
    public function getTemplates(): array;

    /**
     * Gets a specific template.
     *
     * @param string $template_id
     *   The template ID.
     *
     * @return array|null
     *   The template definition or NULL if not found.
     */
    public function getTemplate(string $template_id): ?array;
}
