<?php

namespace Drupal\commerce_shipping_label;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_shipping\Entity\ShipmentInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Utility\Token;

/**
 * Service for generating shipping labels.
 */
class ShippingLabelGenerator implements ShippingLabelGeneratorInterface
{

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected $token;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a new ShippingLabelGenerator.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    Token $token,
    EntityTypeManagerInterface $entity_type_manager,
    RendererInterface $renderer,
    ModuleHandlerInterface $module_handler
  ) {
    $this->configFactory = $config_factory;
    $this->token = $token;
    $this->entityTypeManager = $entity_type_manager;
    $this->renderer = $renderer;
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public function generateFromOrder(OrderInterface $order, ?string $template_id = NULL): string
  {
    $render_array = $this->buildLabelRenderArray($order, $template_id);
    return $this->renderToPdf([$render_array], $template_id);
  }

  /**
   * {@inheritdoc}
   */
  public function generateFromShipment(ShipmentInterface $shipment, ?string $template_id = NULL): string
  {
    $render_array = $this->buildLabelRenderArray($shipment, $template_id);
    return $this->renderToPdf([$render_array], $template_id);
  }

  /**
   * {@inheritdoc}
   */
  public function generateBulk(array $entities, ?string $template_id = NULL): string
  {
    $render_arrays = [];
    foreach ($entities as $entity) {
      $render_arrays[] = $this->buildLabelRenderArray($entity, $template_id);
    }
    return $this->renderToPdf($render_arrays, $template_id);
  }

  /**
   * {@inheritdoc}
   */
  public function buildLabelRenderArray($entity, ?string $template_id = NULL): array
  {
    $template = $this->getActiveTemplate($template_id);

    // Determine if this is an order or shipment.
    $order = NULL;
    $shipment = NULL;

    if ($entity instanceof ShipmentInterface) {
      $shipment = $entity;
      $order = $shipment->getOrder();
    } elseif ($entity instanceof OrderInterface) {
      $order = $entity;
      // Try to get the first shipment if available.
      $shipments = $order->get('shipments')->referencedEntities();
      $shipment = !empty($shipments) ? reset($shipments) : NULL;
    }

    // Build token data.
    $token_data = [
      'commerce_order' => $order,
    ];
    if ($shipment) {
      $token_data['commerce_shipment'] = $shipment;
    }

    // Replace tokens in template content.
    $content = $this->token->replace($template['content'], $token_data, ['clear' => TRUE]);

    return [
      '#theme' => 'commerce_shipping_label',
      '#content' => $content,
      '#order' => $order,
      '#shipment' => $shipment,
      '#template_id' => $template['id'],
      '#width' => $template['width'],
      '#height' => $template['height'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getTemplates(): array
  {
    $config = $this->configFactory->get('commerce_shipping_label.settings');
    $templates = $config->get('templates') ?? [];

    $result = [];
    foreach ($templates as $template) {
      $result[$template['id']] = $template;
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function getTemplate(string $template_id): ?array
  {
    $templates = $this->getTemplates();
    return $templates[$template_id] ?? NULL;
  }

  /**
   * Gets the active template to use.
   *
   * @param string|null $template_id
   *   The requested template ID, or NULL for default.
   *
   * @return array
   *   The template definition.
   */
  protected function getActiveTemplate(?string $template_id = NULL): array
  {
    if ($template_id === NULL) {
      $config = $this->configFactory->get('commerce_shipping_label.settings');
      $template_id = $config->get('default_template') ?? 'standard';
    }

    $template = $this->getTemplate($template_id);

    if (!$template) {
      // Fallback to first available template.
      $templates = $this->getTemplates();
      $template = reset($templates);
    }

    return $template;
  }

  /**
   * Renders label render arrays to PDF.
   *
   * @param array $render_arrays
   *   Array of render arrays.
   * @param string|null $template_id
   *   The template ID for dimensions.
   *
   * @return string
   *   The PDF content.
   */
  protected function renderToPdf(array $render_arrays, ?string $template_id = NULL): string
  {
    $template = $this->getActiveTemplate($template_id);

    // Check if DomPDF is available.
    if (class_exists('\Dompdf\Dompdf')) {
      return $this->renderWithDomPdf($render_arrays, $template);
    }

    // Fallback: render HTML with print-friendly styling.
    return $this->renderAsHtml($render_arrays, $template);
  }

  /**
   * Renders using DomPDF library.
   *
   * @param array $render_arrays
   *   Array of render arrays.
   * @param array $template
   *   The template definition.
   *
   * @return string
   *   The PDF content.
   */
  protected function renderWithDomPdf(array $render_arrays, array $template): string
  {
    // Build HTML content.
    $html = $this->buildHtmlContent($render_arrays, $template);

    // Check if DomPDF is available.
    if (!class_exists('\Dompdf\Dompdf')) {
      // Fallback to HTML if DomPDF is not available.
      return $html;
    }

    // Configure DomPDF options for UTF-8 and Turkish character support.
    $options = new \Dompdf\Options();
    $options->set('isRemoteEnabled', TRUE);
    $options->set('isHtml5ParserEnabled', TRUE);
    $options->set('defaultFont', 'DejaVu Sans');
    $options->set('isFontSubsettingEnabled', TRUE);
    // Use the bundled font directory.
    $options->set('fontDir', sys_get_temp_dir());
    $options->set('fontCache', sys_get_temp_dir());

    // Use DomPDF directly.
    $dompdf = new \Dompdf\Dompdf($options);

    // Set paper size based on template dimensions.
    $width_mm = $template['width'];
    $height_mm = $template['height'];

    // Convert mm to points (1 mm = 2.83465 points).
    $width_pt = $width_mm * 2.83465;
    $height_pt = $height_mm * 2.83465;

    $dompdf->setPaper([0, 0, $width_pt, $height_pt]);
    $dompdf->loadHtml($html, 'UTF-8');
    $dompdf->render();

    return $dompdf->output();
  }

  /**
   * Renders as HTML fallback.
   *
   * @param array $render_arrays
   *   Array of render arrays.
   * @param array $template
   *   The template definition.
   *
   * @return string
   *   The HTML content.
   */
  protected function renderAsHtml(array $render_arrays, array $template): string
  {
    return $this->buildHtmlContent($render_arrays, $template);
  }

  /**
   * Builds complete HTML content with styling.
   *
   * @param array $render_arrays
   *   Array of render arrays.
   * @param array $template
   *   The template definition.
   *
   * @return string
   *   The complete HTML.
   */
  protected function buildHtmlContent(array $render_arrays, array $template): string
  {
    $width_mm = $template['width'];
    $height_mm = $template['height'];
    $template_css = $template['css'] ?? '';

    // Base CSS - minimal required styles.
    $base_css = '
    @page {
      size: ' . $width_mm . 'mm ' . $height_mm . 'mm;
      margin: 2mm;
    }
    * {
      box-sizing: border-box;
      margin: 0;
      padding: 0;
    }
    body {
      font-family: DejaVu Sans, Arial, sans-serif;
      font-size: 9pt;
      line-height: 1.3;
      color: #000;
      margin: 0;
      padding: 0;
    }
    .page-container {
      page-break-after: always;
      page-break-inside: avoid;
    }
    .page-container:last-child {
      page-break-after: auto;
    }
    ';

    // Default CSS if template has no custom CSS.
    if (empty(trim($template_css))) {
      $template_css = '
    .shipping-label {
      padding: 2mm;
    }
    .sender-section, .recipient-section {
      width: 100%;
      table-layout: fixed;
      border: 1px solid #999;
      margin-bottom: 2mm;
    }
    .sender-section td, .recipient-section td {
      padding: 2mm;
      word-wrap: break-word;
    }
    .section-title {
      font-size: 8pt;
      font-weight: bold;
      text-transform: uppercase;
      color: #555;
      margin-bottom: 1mm;
    }
    .sender-name, .recipient-name {
      font-weight: bold;
      font-size: 11pt;
    }
    .sender-address, .recipient-address {
      font-size: 9pt;
    }
    .recipient-phone {
      font-size: 9pt;
      margin-top: 1mm;
    }
    .barcode-section {
      text-align: center;
      padding: 2mm 0;
      border-top: 1px dashed #999;
      border-bottom: 1px dashed #999;
      margin-bottom: 2mm;
    }
    .barcode-section img {
      max-width: 80%;
      height: 40px;
    }
    .order-info, .order-number {
      font-size: 9pt;
      font-weight: bold;
      text-align: center;
    }
    .sender-compact {
      font-size: 8pt;
      padding: 2mm;
      margin-bottom: 2mm;
      border-bottom: 1px solid #000;
      background-color: #f0f0f0;
    }
      ';
    }

    $html = '<!DOCTYPE html>
<html>
<head>
  <meta http-equiv="Content-Type" content="text/html; charset=utf-8"/>
  <title>Shipping Label</title>
  <style>' . $base_css . $template_css . '</style>
</head>
<body>';

    foreach ($render_arrays as $render_array) {
      $content = (string) $this->renderer->renderRoot($render_array);
      // Trim whitespace to prevent blank pages.
      $content = trim($content);
      $html .= '<div class="page-container">' . $content . '</div>';
    }

    $html .= '</body></html>';

    return $html;
  }
}
