<?php

namespace Drupal\commerce_shipping_label\Plugin\Action;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Url;
use Drupal\views_bulk_operations\Action\ViewsBulkOperationsActionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Generates shipping labels for selected orders.
 *
 * @Action(
 *   id = "commerce_shipping_label_generate",
 *   label = @Translation("Generate Shipping Labels"),
 *   type = "commerce_order",
 *   confirm = TRUE,
 * )
 */
class GenerateShippingLabelAction extends ViewsBulkOperationsActionBase implements PluginFormInterface, ContainerFactoryPluginInterface
{

    /**
     * The temp store factory.
     *
     * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
     */
    protected $tempStoreFactory;

    /**
     * The label generator service.
     *
     * @var \Drupal\commerce_shipping_label\ShippingLabelGeneratorInterface
     */
    protected $labelGenerator;

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected $configFactory;

    /**
     * The messenger.
     *
     * @var \Drupal\Core\Messenger\MessengerInterface
     */
    protected $messenger;

    /**
     * Collected order IDs for bulk processing.
     *
     * @var array
     */
    protected static $collectedOrderIds = [];

    /**
     * Constructs a new GenerateShippingLabelAction.
     */
    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        PrivateTempStoreFactory $temp_store_factory,
        $config_factory,
        $messenger
    ) {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->tempStoreFactory = $temp_store_factory;
        $this->configFactory = $config_factory;
        $this->messenger = $messenger;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('tempstore.private'),
            $container->get('config.factory'),
            $container->get('messenger')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function execute($entity = NULL)
    {
        if (!$entity instanceof OrderInterface) {
            return;
        }

        // Accumulate order IDs in temp store (persists across batch requests).
        $temp_store = $this->tempStoreFactory->get('commerce_shipping_label');
        $order_ids = $temp_store->get('bulk_order_ids') ?? [];
        $order_ids[] = $entity->id();
        $temp_store->set('bulk_order_ids', array_unique($order_ids));
        $temp_store->set('bulk_template_id', $this->configuration['template_id'] ?? NULL);

        return $this->t('Added order @order to label batch.', [
            '@order' => $entity->getOrderNumber() ?? $entity->id(),
        ]);
    }

    /**
     * {@inheritdoc}
     */
    public function executeMultiple(array $entities)
    {
        $order_ids = [];

        foreach ($entities as $entity) {
            if ($entity instanceof OrderInterface) {
                $order_ids[] = $entity->id();
            }
        }

        if (empty($order_ids)) {
            $this->messenger->addError($this->t('No valid orders selected.'));
            return [];
        }

        // Store order IDs in temp store for the download controller.
        $temp_store = $this->tempStoreFactory->get('commerce_shipping_label');
        $temp_store->set('bulk_order_ids', $order_ids);
        $temp_store->set('bulk_template_id', $this->configuration['template_id'] ?? NULL);

        // Set redirect to bulk download - works with confirm=TRUE.
        $this->context['redirect_url'] = Url::fromRoute('commerce_shipping_label.bulk_download');

        return [];
    }

    /**
     * {@inheritdoc}
     */
    public function buildConfigurationForm(array $form, FormStateInterface $form_state)
    {
        // Get available templates.
        $config = $this->configFactory->get('commerce_shipping_label.settings');
        $templates = $config->get('templates') ?? [];
        $options = [];

        foreach ($templates as $template) {
            $options[$template['id']] = $template['label'];
        }

        $form['template_id'] = [
            '#type' => 'select',
            '#title' => $this->t('Label Template'),
            '#description' => $this->t('Select the template to use for generating labels.'),
            '#options' => $options,
            '#default_value' => $config->get('default_template') ?? 'standard',
            '#required' => TRUE,
        ];

        return $form;
    }

    /**
     * {@inheritdoc}
     */
    public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void
    {
        // No additional validation needed.
    }

    /**
     * {@inheritdoc}
     */
    public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void
    {
        $this->configuration['template_id'] = $form_state->getValue('template_id');
    }

    /**
     * {@inheritdoc}
     */
    public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE)
    {
        if (!$object instanceof OrderInterface) {
            $result = AccessResult::forbidden();
            return $return_as_object ? $result : $result->isAllowed();
        }

        $result = AccessResult::allowedIfHasPermission($account, 'generate commerce shipping labels');

        return $return_as_object ? $result : $result->isAllowed();
    }

    /**
     * {@inheritdoc}
     */
    public static function finished($success, array $results, array $operations): ?RedirectResponse
    {
        if ($success) {
            // Add download link message.
            $download_url = Url::fromRoute('commerce_shipping_label.bulk_download')->toString();
            \Drupal::messenger()->addStatus(t('Shipping labels generated. <a href="@url" id="shipping-label-download">Click here to download the PDF</a>', [
                '@url' => $download_url,
            ]));

            // Add JavaScript to auto-trigger download.
            // Using a small inline script that will be processed on the next page load.
            $_SESSION['commerce_shipping_label_auto_download'] = $download_url;
        }

        // Redirect back to orders view.
        $orders_url = Url::fromRoute('entity.commerce_order.collection')->toString();
        return new RedirectResponse($orders_url);
    }
}
