<?php

namespace Drupal\commerce_shipping_label\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Settings form for shipping labels.
 */
class ShippingLabelSettingsForm extends ConfigFormBase
{

    /**
     * {@inheritdoc}
     */
    public function getFormId()
    {
        return 'commerce_shipping_label_settings';
    }

    /**
     * {@inheritdoc}
     */
    protected function getEditableConfigNames()
    {
        return ['commerce_shipping_label.settings'];
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(array $form, FormStateInterface $form_state)
    {
        $config = $this->config('commerce_shipping_label.settings');

        // Template selection.
        $templates = $config->get('templates') ?? [];
        $template_options = [];
        foreach ($templates as $template) {
            $template_options[$template['id']] = $template['label'];
        }

        $form['default_template'] = [
            '#type' => 'select',
            '#title' => $this->t('Default Template'),
            '#description' => $this->t('Select the default template for shipping labels.'),
            '#options' => $template_options,
            '#default_value' => $config->get('default_template') ?? 'standard',
        ];

        // Sender override section.
        $form['sender_override'] = [
            '#type' => 'details',
            '#title' => $this->t('Sender Address Override'),
            '#description' => $this->t('Override the store address with a custom sender address. If disabled, the store address will be used.'),
            '#open' => !empty($config->get('sender_override.enabled')),
            '#tree' => TRUE,
        ];

        $form['sender_override']['enabled'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Enable sender address override'),
            '#default_value' => $config->get('sender_override.enabled') ?? FALSE,
        ];

        $states_visible = [
            'visible' => [
                ':input[name="sender_override[enabled]"]' => ['checked' => TRUE],
            ],
        ];

        $form['sender_override']['name'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Sender Name'),
            '#default_value' => $config->get('sender_override.name') ?? '',
            '#states' => $states_visible,
        ];

        $form['sender_override']['address_line1'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Street Address'),
            '#default_value' => $config->get('sender_override.address_line1') ?? '',
            '#states' => $states_visible,
        ];

        $form['sender_override']['address_line2'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Street Address 2'),
            '#default_value' => $config->get('sender_override.address_line2') ?? '',
            '#states' => $states_visible,
        ];

        $form['sender_override']['dependent_locality'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Neighbourhood'),
            '#default_value' => $config->get('sender_override.dependent_locality') ?? '',
            '#states' => $states_visible,
        ];

        $form['sender_override']['locality'] = [
            '#type' => 'textfield',
            '#title' => $this->t('District'),
            '#default_value' => $config->get('sender_override.locality') ?? '',
            '#states' => $states_visible,
        ];

        $form['sender_override']['administrative_area'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Province'),
            '#description' => $this->t('Province code (e.g., TR-34 for İstanbul)'),
            '#default_value' => $config->get('sender_override.administrative_area') ?? '',
            '#states' => $states_visible,
        ];

        $form['sender_override']['postal_code'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Postal Code'),
            '#default_value' => $config->get('sender_override.postal_code') ?? '',
            '#size' => 10,
            '#states' => $states_visible,
        ];

        $form['sender_override']['country_code'] = [
            '#type' => 'hidden',
            '#value' => 'TR',
        ];

        // Templates section.
        $form['templates'] = [
            '#type' => 'details',
            '#title' => $this->t('Label Templates'),
            '#open' => TRUE,
        ];

        $form['templates']['templates_container'] = [
            '#type' => 'container',
            '#prefix' => '<div id="templates-wrapper">',
            '#suffix' => '</div>',
        ];

        // Get templates from form state or config.
        $templates_values = $form_state->get('templates');
        if ($templates_values === NULL) {
            $templates_values = $config->get('templates') ?? [];
            $form_state->set('templates', $templates_values);
        }

        foreach ($templates_values as $index => $template) {
            $form['templates']['templates_container'][$index] = [
                '#type' => 'details',
                '#title' => $template['label'] ?? $this->t('New Template'),
                '#open' => FALSE,
            ];

            $form['templates']['templates_container'][$index]['label'] = [
                '#type' => 'textfield',
                '#title' => $this->t('Template Name'),
                '#default_value' => $template['label'] ?? '',
                '#required' => TRUE,
                '#id' => 'edit-templates-templates-container-' . $index . '-label',
            ];

            $form['templates']['templates_container'][$index]['id'] = [
                '#type' => 'machine_name',
                '#title' => $this->t('Template ID'),
                '#default_value' => $template['id'] ?? '',
                '#machine_name' => [
                    'exists' => [$this, 'templateExists'],
                    'source' => ['templates', 'templates_container', $index, 'label'],
                ],
                '#disabled' => !empty($template['id']),
            ];

            $form['templates']['templates_container'][$index]['dimensions'] = [
                '#type' => 'container',
                '#attributes' => ['class' => ['container-inline']],
            ];

            $form['templates']['templates_container'][$index]['dimensions']['width'] = [
                '#type' => 'number',
                '#title' => $this->t('Width (mm)'),
                '#default_value' => $template['width'] ?? 101.6,
                '#min' => 20,
                '#max' => 300,
                '#step' => 0.1,
            ];

            $form['templates']['templates_container'][$index]['dimensions']['height'] = [
                '#type' => 'number',
                '#title' => $this->t('Height (mm)'),
                '#default_value' => $template['height'] ?? 152.4,
                '#min' => 20,
                '#max' => 300,
                '#step' => 0.1,
            ];

            $form['templates']['templates_container'][$index]['content'] = [
                '#type' => 'textarea',
                '#title' => $this->t('Template Content'),
                '#description' => $this->t('HTML content with tokens. Available tokens are listed below.'),
                '#default_value' => $template['content'] ?? '',
                '#rows' => 10,
            ];

            $form['templates']['templates_container'][$index]['css'] = [
                '#type' => 'textarea',
                '#title' => $this->t('Template CSS'),
                '#description' => $this->t('Custom CSS styles for this template. These will be included in the PDF.'),
                '#default_value' => $template['css'] ?? '',
                '#rows' => 10,
            ];

            $form['templates']['templates_container'][$index]['remove'] = [
                '#type' => 'submit',
                '#value' => $this->t('Remove @label', ['@label' => $template['label'] ?? 'template']),
                '#name' => 'remove_' . $index,
                '#submit' => ['::removeTemplate'],
                '#ajax' => [
                    'callback' => '::templatesCallback',
                    'wrapper' => 'templates-wrapper',
                ],
                '#limit_validation_errors' => [],
            ];
        }

        $form['templates']['add_template'] = [
            '#type' => 'submit',
            '#value' => $this->t('Add Template'),
            '#submit' => ['::addTemplate'],
            '#ajax' => [
                'callback' => '::templatesCallback',
                'wrapper' => 'templates-wrapper',
            ],
            '#limit_validation_errors' => [],
        ];

        // Available tokens reference.
        $form['tokens_help'] = [
            '#type' => 'details',
            '#title' => $this->t('Available Tokens'),
            '#open' => FALSE,
        ];

        $form['tokens_help']['list'] = [
            '#type' => 'item',
            '#markup' => '
        <h4>Sender Tokens:</h4>
        <ul>
          <li><code>[shipping_label:sender:name]</code> - Store/Sender name</li>
          <li><code>[shipping_label:sender:full_address]</code> - Complete formatted address</li>
          <li><code>[shipping_label:sender:address_line1]</code> - Address line 1</li>
          <li><code>[shipping_label:sender:locality]</code> - City</li>
          <li><code>[shipping_label:sender:postal_code]</code> - Postal code</li>
        </ul>
        <h4>Recipient Tokens:</h4>
        <ul>
          <li><code>[shipping_label:recipient:name]</code> - Recipient name</li>
          <li><code>[shipping_label:recipient:full_address]</code> - Complete formatted address</li>
          <li><code>[shipping_label:recipient:phone]</code> - Phone number</li>
          <li><code>[shipping_label:recipient:locality]</code> - City</li>
          <li><code>[shipping_label:recipient:postal_code]</code> - Postal code</li>
        </ul>
        <h4>Barcode Tokens:</h4>
        <ul>
          <li><code>[shipping_label:order_barcode]</code> - Order number as barcode image</li>
          <li><code>[shipping_label:tracking_barcode]</code> - Tracking code as barcode image</li>
        </ul>
        <h4>Order Tokens:</h4>
        <ul>
          <li><code>[commerce_order:order_number]</code> - Order number</li>
          <li><code>[commerce_order:created:medium]</code> - Order date</li>
        </ul>
        <h4>Shipment Tokens:</h4>
        <ul>
          <li><code>[commerce_shipment:tracking_code]</code> - Tracking code</li>
        </ul>
      ',
        ];

        return parent::buildForm($form, $form_state);
    }

    /**
     * Ajax callback for templates section.
     */
    public function templatesCallback(array &$form, FormStateInterface $form_state)
    {
        return $form['templates']['templates_container'];
    }

    /**
     * Submit handler to add a template.
     */
    public function addTemplate(array &$form, FormStateInterface $form_state)
    {
        $templates = $form_state->get('templates') ?? [];
        $templates[] = [
            'id' => '',
            'label' => 'New Template',
            'width' => 101.6,
            'height' => 152.4,
            'content' => '',
        ];
        $form_state->set('templates', $templates);
        $form_state->setRebuild();
    }

    /**
     * Submit handler to remove a template.
     */
    public function removeTemplate(array &$form, FormStateInterface $form_state)
    {
        $triggering_element = $form_state->getTriggeringElement();
        $button_name = $triggering_element['#name'];

        if (preg_match('/^remove_(\d+)$/', $button_name, $matches)) {
            $index = $matches[1];
            $templates = $form_state->get('templates') ?? [];
            unset($templates[$index]);
            $templates = array_values($templates);
            $form_state->set('templates', $templates);
        }

        $form_state->setRebuild();
    }

    /**
     * Checks if a template ID already exists.
     *
     * Note: For existing templates, the ID field is disabled so this
     * validation only applies to new templates. We return FALSE here
     * to avoid blocking re-saves of existing configs.
     */
    public function templateExists($id)
    {
        // Don't validate empty IDs.
        if (empty($id)) {
            return FALSE;
        }

        // For existing templates (disabled fields), don't trigger validation error.
        // The machine_name field is disabled for existing templates anyway.
        $config = $this->config('commerce_shipping_label.settings');
        $templates = $config->get('templates') ?? [];

        foreach ($templates as $template) {
            if ($template['id'] === $id) {
                // This is an existing template being re-saved, not a duplicate.
                return FALSE;
            }
        }

        return FALSE;
    }

    /**
     * {@inheritdoc}
     */
    public function submitForm(array &$form, FormStateInterface $form_state)
    {
        $config = $this->config('commerce_shipping_label.settings');

        // Save default template.
        $config->set('default_template', $form_state->getValue('default_template'));

        // Save sender override.
        $sender_override = $form_state->getValue('sender_override') ?? [];

        $config->set('sender_override', [
            'enabled' => !empty($sender_override['enabled']),
            'name' => $sender_override['name'] ?? '',
            'address_line1' => $sender_override['address_line1'] ?? '',
            'address_line2' => $sender_override['address_line2'] ?? '',
            'locality' => $sender_override['locality'] ?? '',
            'dependent_locality' => $sender_override['dependent_locality'] ?? '',
            'administrative_area' => $sender_override['administrative_area'] ?? '',
            'postal_code' => $sender_override['postal_code'] ?? '',
            'country_code' => $sender_override['country_code'] ?? 'TR',
        ]);

        // Save templates - use form state templates if available, otherwise preserve existing.
        $templates_container = $form_state->getValue(['templates', 'templates_container']);
        $templates = [];

        if (is_array($templates_container) && !empty($templates_container)) {
            foreach ($templates_container as $index => $values) {
                if (!is_array($values) || empty($values['id'])) {
                    continue;
                }
                $templates[] = [
                    'id' => $values['id'],
                    'label' => $values['label'],
                    'width' => (float) ($values['dimensions']['width'] ?? $values['width'] ?? 101.6),
                    'height' => (float) ($values['dimensions']['height'] ?? $values['height'] ?? 152.4),
                    'content' => $values['content'] ?? '',
                    'css' => $values['css'] ?? '',
                ];
            }
        }

        // Only update templates if we have valid data, otherwise preserve existing.
        if (!empty($templates)) {
            $config->set('templates', $templates);
        }

        $config->save();

        parent::submitForm($form, $form_state);
    }
}
