<?php

namespace Drupal\commerce_shipping_label\Controller;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_shipping\Entity\ShipmentInterface;
use Drupal\commerce_shipping_label\ShippingLabelGeneratorInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for shipping label downloads.
 */
class ShippingLabelController extends ControllerBase
{

    /**
     * The label generator service.
     *
     * @var \Drupal\commerce_shipping_label\ShippingLabelGeneratorInterface
     */
    protected $labelGenerator;

    /**
     * The private temp store factory.
     *
     * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
     */
    protected $tempStoreFactory;

    /**
     * Constructs a new ShippingLabelController.
     *
     * @param \Drupal\commerce_shipping_label\ShippingLabelGeneratorInterface $label_generator
     *   The label generator service.
     * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
     *   The private temp store factory.
     */
    public function __construct(
        ShippingLabelGeneratorInterface $label_generator,
        PrivateTempStoreFactory $temp_store_factory
    ) {
        $this->labelGenerator = $label_generator;
        $this->tempStoreFactory = $temp_store_factory;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container)
    {
        return new static(
            $container->get('commerce_shipping_label.generator'),
            $container->get('tempstore.private')
        );
    }

    /**
     * Downloads a shipping label for an order.
     *
     * @param \Drupal\commerce_order\Entity\OrderInterface $commerce_order
     *   The order entity.
     * @param \Symfony\Component\HttpFoundation\Request $request
     *   The request object.
     *
     * @return \Symfony\Component\HttpFoundation\Response
     *   The PDF response.
     */
    public function downloadOrderLabel(OrderInterface $commerce_order, Request $request): Response
    {
        $template_id = $request->query->get('template');

        try {
            $content = $this->labelGenerator->generateFromOrder($commerce_order, $template_id);
            $filename = 'shipping-label-order-' . ($commerce_order->getOrderNumber() ?? $commerce_order->id()) . '.pdf';

            return $this->createPdfResponse($content, $filename);
        } catch (\Exception $e) {
            $this->messenger()->addError($this->t('Failed to generate label: @message', ['@message' => $e->getMessage()]));
            return $this->redirect('entity.commerce_order.canonical', ['commerce_order' => $commerce_order->id()]);
        }
    }

    /**
     * Downloads a shipping label for a shipment.
     *
     * @param \Drupal\commerce_order\Entity\OrderInterface $commerce_order
     *   The order entity.
     * @param \Drupal\commerce_shipping\Entity\ShipmentInterface $commerce_shipment
     *   The shipment entity.
     * @param \Symfony\Component\HttpFoundation\Request $request
     *   The request object.
     *
     * @return \Symfony\Component\HttpFoundation\Response
     *   The PDF response.
     */
    public function downloadShipmentLabel(
        OrderInterface $commerce_order,
        ShipmentInterface $commerce_shipment,
        Request $request
    ): Response {
        $template_id = $request->query->get('template');

        try {
            $content = $this->labelGenerator->generateFromShipment($commerce_shipment, $template_id);
            $filename = 'shipping-label-shipment-' . $commerce_shipment->id() . '.pdf';

            return $this->createPdfResponse($content, $filename);
        } catch (\Exception $e) {
            $this->messenger()->addError($this->t('Failed to generate label: @message', ['@message' => $e->getMessage()]));
            return $this->redirect('entity.commerce_order.canonical', ['commerce_order' => $commerce_order->id()]);
        }
    }

    /**
     * Downloads bulk shipping labels.
     *
     * @param \Symfony\Component\HttpFoundation\Request $request
     *   The request object.
     *
     * @return \Symfony\Component\HttpFoundation\Response
     *   The PDF response.
     */
    public function downloadBulk(Request $request): Response
    {
        $temp_store = $this->tempStoreFactory->get('commerce_shipping_label');
        $order_ids = $temp_store->get('bulk_order_ids');
        $template_id = $temp_store->get('bulk_template_id');
        $access_time = $temp_store->get('bulk_access_time');

        if (empty($order_ids)) {
            $this->messenger()->addError($this->t('No orders selected for label generation.'));
            return $this->redirect('entity.commerce_order.collection');
        }

        try {
            $orders = $this->entityTypeManager()->getStorage('commerce_order')->loadMultiple($order_ids);
            $content = $this->labelGenerator->generateBulk($orders, $template_id);
            $filename = 'shipping-labels-bulk-' . date('Y-m-d-His') . '.pdf';

            // Set access time on first download, clear after 60 seconds.
            if (empty($access_time)) {
                $temp_store->set('bulk_access_time', time());
            } elseif (time() - $access_time > 60) {
                // Clear temp store after 60 seconds.
                $temp_store->delete('bulk_order_ids');
                $temp_store->delete('bulk_template_id');
                $temp_store->delete('bulk_access_time');
            }

            return $this->createPdfResponse($content, $filename);
        } catch (\Exception $e) {
            $this->messenger()->addError($this->t('Failed to generate labels: @message', ['@message' => $e->getMessage()]));
            return $this->redirect('entity.commerce_order.collection');
        }
    }

    /**
     * Creates a PDF response.
     *
     * @param string $content
     *   The PDF content.
     * @param string $filename
     *   The filename.
     *
     * @return \Symfony\Component\HttpFoundation\Response
     *   The response.
     */
    protected function createPdfResponse(string $content, string $filename): Response
    {
        // Check if content is actually PDF or HTML.
        $is_pdf = strpos($content, '%PDF') === 0;

        if ($is_pdf) {
            $content_type = 'application/pdf';
        } else {
            // If Entity Print is not available, we return HTML.
            $content_type = 'text/html';
            $filename = str_replace('.pdf', '.html', $filename);
        }

        $response = new Response($content);
        $response->headers->set('Content-Type', $content_type);
        $response->headers->set('Content-Disposition', 'attachment; filename="' . $filename . '"');
        $response->headers->set('Content-Length', strlen($content));

        return $response;
    }
}
