<?php

namespace Drupal\commerce_shipping_label;

/**
 * Service for generating barcodes.
 */
class BarcodeGenerator
{

    /**
     * Generates a Code128 barcode as an SVG.
     *
     * @param string $data
     *   The data to encode.
     * @param int $width
     *   The width of the barcode.
     * @param int $height
     *   The height of the barcode.
     *
     * @return string
     *   The SVG barcode markup.
     */
    public function generateCode128(string $data, int $width = 200, int $height = 50): string
    {
        // Code128 encoding patterns.
        $code128_patterns = $this->getCode128Patterns();

        // Clean input - only alphanumeric and some special chars.
        $data = preg_replace('/[^A-Za-z0-9\-\.]/', '', $data);

        if (empty($data)) {
            return '';
        }

        // Encode using Code128B (supports full ASCII).
        $encoded = $this->encodeCode128B($data, $code128_patterns);

        if (empty($encoded)) {
            return '';
        }

        // Generate SVG.
        return $this->generateSvg($encoded, $data, $width, $height);
    }

    /**
     * Generates a barcode as a data URI for embedding in HTML.
     *
     * @param string $data
     *   The data to encode.
     * @param int $width
     *   The width of the barcode.
     * @param int $height
     *   The height of the barcode.
     *
     * @return string
     *   An img tag with data URI.
     */
    public function generateBarcodeImg(string $data, int $width = 200, int $height = 50): string
    {
        $svg = $this->generateCode128($data, $width, $height);

        if (empty($svg)) {
            return '<span class="barcode-error">Invalid barcode data</span>';
        }

        $data_uri = 'data:image/svg+xml;base64,' . base64_encode($svg);

        return '<img src="' . $data_uri . '" alt="Barcode: ' . htmlspecialchars($data) . '" class="barcode-image" />';
    }

    /**
     * Encodes data using Code128B.
     *
     * @param string $data
     *   The data to encode.
     * @param array $patterns
     *   The Code128 patterns.
     *
     * @return string
     *   Binary pattern string.
     */
    protected function encodeCode128B(string $data, array $patterns): string
    {
        // Start code B.
        $checksum = 104;
        $encoded = $patterns[104];

        $chars = str_split($data);
        foreach ($chars as $index => $char) {
            $code = ord($char) - 32;
            if ($code < 0 || $code > 95) {
                // Invalid character for Code128B.
                continue;
            }
            $encoded .= $patterns[$code];
            $checksum += $code * ($index + 1);
        }

        // Add checksum.
        $checksum = $checksum % 103;
        $encoded .= $patterns[$checksum];

        // Add stop code.
        $encoded .= $patterns[106];

        return $encoded;
    }

    /**
     * Generates SVG from binary pattern.
     *
     * @param string $pattern
     *   Binary pattern string (1s and 0s).
     * @param string $data
     *   Original data for display.
     * @param int $width
     *   Target width.
     * @param int $height
     *   Target height.
     *
     * @return string
     *   SVG markup.
     */
    protected function generateSvg(string $pattern, string $data, int $width, int $height): string
    {
        $pattern_length = strlen($pattern);
        if ($pattern_length === 0) {
            return '';
        }

        $bar_width = $width / $pattern_length;
        $bar_height = $height - 15; // Leave space for text.

        $svg = '<svg xmlns="http://www.w3.org/2000/svg" width="' . $width . '" height="' . $height . '" viewBox="0 0 ' . $width . ' ' . $height . '">';
        $svg .= '<rect width="100%" height="100%" fill="white"/>';

        $x = 0;
        for ($i = 0; $i < $pattern_length; $i++) {
            if ($pattern[$i] === '1') {
                $svg .= '<rect x="' . ($x * $bar_width) . '" y="0" width="' . $bar_width . '" height="' . $bar_height . '" fill="black"/>';
            }
            $x++;
        }

        // Add text below barcode.
        $svg .= '<text x="' . ($width / 2) . '" y="' . ($height - 2) . '" text-anchor="middle" font-family="monospace" font-size="10">' . htmlspecialchars($data) . '</text>';
        $svg .= '</svg>';

        return $svg;
    }

    /**
     * Gets Code128 patterns.
     *
     * @return array
     *   Array of binary patterns indexed by code value.
     */
    protected function getCode128Patterns(): array
    {
        return [
            0 => '11011001100',
            1 => '11001101100',
            2 => '11001100110',
            3 => '10010011000',
            4 => '10010001100',
            5 => '10001001100',
            6 => '10011001000',
            7 => '10011000100',
            8 => '10001100100',
            9 => '11001001000',
            10 => '11001000100',
            11 => '11000100100',
            12 => '10110011100',
            13 => '10011011100',
            14 => '10011001110',
            15 => '10111001100',
            16 => '10011101100',
            17 => '10011100110',
            18 => '11001110010',
            19 => '11001011100',
            20 => '11001001110',
            21 => '11011100100',
            22 => '11001110100',
            23 => '11101101110',
            24 => '11101001100',
            25 => '11100101100',
            26 => '11100100110',
            27 => '11101100100',
            28 => '11100110100',
            29 => '11100110010',
            30 => '11011011000',
            31 => '11011000110',
            32 => '11000110110',
            33 => '10100011000',
            34 => '10001011000',
            35 => '10001000110',
            36 => '10110001000',
            37 => '10001101000',
            38 => '10001100010',
            39 => '11010001000',
            40 => '11000101000',
            41 => '11000100010',
            42 => '10110111000',
            43 => '10110001110',
            44 => '10001101110',
            45 => '10111011000',
            46 => '10111000110',
            47 => '10001110110',
            48 => '11101110110',
            49 => '11010001110',
            50 => '11000101110',
            51 => '11011101000',
            52 => '11011100010',
            53 => '11011101110',
            54 => '11101011000',
            55 => '11101000110',
            56 => '11100010110',
            57 => '11101101000',
            58 => '11101100010',
            59 => '11100011010',
            60 => '11101111010',
            61 => '11001000010',
            62 => '11110001010',
            63 => '10100110000',
            64 => '10100001100',
            65 => '10010110000',
            66 => '10010000110',
            67 => '10000101100',
            68 => '10000100110',
            69 => '10110010000',
            70 => '10110000100',
            71 => '10011010000',
            72 => '10011000010',
            73 => '10000110100',
            74 => '10000110010',
            75 => '11000010010',
            76 => '11001010000',
            77 => '11110111010',
            78 => '11000010100',
            79 => '10001111010',
            80 => '10100111100',
            81 => '10010111100',
            82 => '10010011110',
            83 => '10111100100',
            84 => '10011110100',
            85 => '10011110010',
            86 => '11110100100',
            87 => '11110010100',
            88 => '11110010010',
            89 => '11011011110',
            90 => '11011110110',
            91 => '11110110110',
            92 => '10101111000',
            93 => '10100011110',
            94 => '10001011110',
            95 => '10111101000',
            96 => '10111100010',
            97 => '11110101000',
            98 => '11110100010',
            99 => '10111011110',
            100 => '10111101110',
            101 => '11101011110',
            102 => '11110101110',
            103 => '11010000100', // Start A
            104 => '11010010000', // Start B
            105 => '11010011100', // Start C
            106 => '1100011101011', // Stop
        ];
    }
}
