<?php

/**
 * @file
 * Provides tokens for shipping labels.
 */

use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\Render\Markup;

/**
 * Implements hook_token_info().
 */
function commerce_shipping_label_token_info()
{
    $info = [];

    // Define token type.
    $info['types']['shipping_label'] = [
        'name' => t('Shipping Label'),
        'description' => t('Tokens related to shipping labels.'),
    ];

    // Sender tokens.
    $info['tokens']['shipping_label']['sender:name'] = [
        'name' => t('Sender Name'),
        'description' => t('The sender/store name.'),
    ];
    $info['tokens']['shipping_label']['sender:address_line1'] = [
        'name' => t('Sender Address Line 1'),
        'description' => t('The first line of the sender address.'),
    ];
    $info['tokens']['shipping_label']['sender:address_line2'] = [
        'name' => t('Sender Address Line 2'),
        'description' => t('The second line of the sender address.'),
    ];
    $info['tokens']['shipping_label']['sender:locality'] = [
        'name' => t('Sender City'),
        'description' => t('The sender city/locality.'),
    ];
    $info['tokens']['shipping_label']['sender:dependent_locality'] = [
        'name' => t('Sender District'),
        'description' => t('The sender district/neighborhood (dependent locality).'),
    ];
    $info['tokens']['shipping_label']['sender:administrative_area'] = [
        'name' => t('Sender State/Province'),
        'description' => t('The sender state or province.'),
    ];
    $info['tokens']['shipping_label']['sender:postal_code'] = [
        'name' => t('Sender Postal Code'),
        'description' => t('The sender postal code.'),
    ];
    $info['tokens']['shipping_label']['sender:country'] = [
        'name' => t('Sender Country'),
        'description' => t('The sender country.'),
    ];
    $info['tokens']['shipping_label']['sender:full_address'] = [
        'name' => t('Sender Full Address'),
        'description' => t('The complete formatted sender address.'),
    ];
    $info['tokens']['shipping_label']['sender:full_local_address'] = [
        'name' => t('Sender Full Local Address'),
        'description' => t('Full Turkish-style address: District, Neighborhood, Street Address.'),
    ];

    // Recipient tokens.
    $info['tokens']['shipping_label']['recipient:name'] = [
        'name' => t('Recipient Name'),
        'description' => t('The recipient name.'),
    ];
    $info['tokens']['shipping_label']['recipient:address_line1'] = [
        'name' => t('Recipient Address Line 1'),
        'description' => t('The first line of the recipient address.'),
    ];
    $info['tokens']['shipping_label']['recipient:address_line2'] = [
        'name' => t('Recipient Address Line 2'),
        'description' => t('The second line of the recipient address.'),
    ];
    $info['tokens']['shipping_label']['recipient:locality'] = [
        'name' => t('Recipient City'),
        'description' => t('The recipient city/locality.'),
    ];
    $info['tokens']['shipping_label']['recipient:dependent_locality'] = [
        'name' => t('Recipient District'),
        'description' => t('The recipient district/neighborhood (dependent locality).'),
    ];
    $info['tokens']['shipping_label']['recipient:administrative_area'] = [
        'name' => t('Recipient State/Province'),
        'description' => t('The recipient state or province.'),
    ];
    $info['tokens']['shipping_label']['recipient:postal_code'] = [
        'name' => t('Recipient Postal Code'),
        'description' => t('The recipient postal code.'),
    ];
    $info['tokens']['shipping_label']['recipient:country'] = [
        'name' => t('Recipient Country'),
        'description' => t('The recipient country.'),
    ];
    $info['tokens']['shipping_label']['recipient:full_address'] = [
        'name' => t('Recipient Full Address'),
        'description' => t('The complete formatted recipient address.'),
    ];
    $info['tokens']['shipping_label']['recipient:full_local_address'] = [
        'name' => t('Recipient Full Local Address'),
        'description' => t('Full Turkish-style address: District, Neighborhood, Street Address.'),
    ];
    $info['tokens']['shipping_label']['recipient:phone'] = [
        'name' => t('Recipient Phone'),
        'description' => t('The recipient phone number.'),
    ];

    // Barcode tokens.
    $info['tokens']['shipping_label']['order_barcode'] = [
        'name' => t('Order Number Barcode'),
        'description' => t('The order number as a barcode image.'),
    ];
    $info['tokens']['shipping_label']['tracking_barcode'] = [
        'name' => t('Tracking Code Barcode'),
        'description' => t('The tracking code as a barcode image.'),
    ];

    return $info;
}

/**
 * Implements hook_tokens().
 */
function commerce_shipping_label_tokens($type, $tokens, array $data, array $options, BubbleableMetadata $bubbleable_metadata)
{
    $replacements = [];

    if ($type !== 'shipping_label') {
        return $replacements;
    }

    // Get order and shipment from data.
    $order = $data['commerce_order'] ?? NULL;
    $shipment = $data['commerce_shipment'] ?? NULL;

    // Get sender info (from store or override settings).
    $sender = _commerce_shipping_label_get_sender_info($order);

    // Get recipient info (from shipping profile).
    $recipient = _commerce_shipping_label_get_recipient_info($order, $shipment);

    foreach ($tokens as $name => $original) {
        // Sender tokens.
        if (strpos($name, 'sender:') === 0) {
            $field = substr($name, 7);
            if (isset($sender[$field])) {
                $replacements[$original] = $sender[$field];
            } else {
                $replacements[$original] = '';
            }
        }

        // Recipient tokens.
        if (strpos($name, 'recipient:') === 0) {
            $field = substr($name, 10);
            if (isset($recipient[$field])) {
                $replacements[$original] = $recipient[$field];
            } else {
                $replacements[$original] = '';
            }
        }

        // Barcode tokens - wrap in Markup to prevent HTML escaping.
        if ($name === 'order_barcode' && $order) {
            /** @var \Drupal\commerce_shipping_label\BarcodeGenerator $barcode_generator */
            $barcode_generator = \Drupal::service('commerce_shipping_label.barcode_generator');
            $order_number = $order->getOrderNumber() ?? $order->id();
            $barcode_html = $barcode_generator->generateBarcodeImg($order_number);
            $replacements[$original] = Markup::create($barcode_html);
        }

        if ($name === 'tracking_barcode' && $shipment) {
            $tracking_code = $shipment->getTrackingCode();
            if ($tracking_code) {
                /** @var \Drupal\commerce_shipping_label\BarcodeGenerator $barcode_generator */
                $barcode_generator = \Drupal::service('commerce_shipping_label.barcode_generator');
                $barcode_html = $barcode_generator->generateBarcodeImg($tracking_code);
                $replacements[$original] = Markup::create($barcode_html);
            } else {
                $replacements[$original] = '';
            }
        }
    }

    return $replacements;
}

/**
 * Gets sender information from store or settings override.
 *
 * @param \Drupal\commerce_order\Entity\OrderInterface|null $order
 *   The order entity.
 *
 * @return array
 *   Array of sender information.
 */
function _commerce_shipping_label_get_sender_info($order): array
{
    $config = \Drupal::config('commerce_shipping_label.settings');
    $sender_override = $config->get('sender_override') ?? [];

    // Check if override is enabled.
    if (!empty($sender_override['enabled'])) {
        $address_parts = array_filter([
            $sender_override['address_line1'] ?? '',
            $sender_override['address_line2'] ?? '',
        ]);

        // Get raw values.
        $administrative_area = $sender_override['administrative_area'] ?? '';
        $locality = $sender_override['locality'] ?? '';
        $dependent_locality = $sender_override['dependent_locality'] ?? '';
        $country_code = $sender_override['country_code'] ?? 'TR';

        // For Turkish addresses, resolve codes to labels.
        if ($country_code === 'TR' && \Drupal::hasService('address_tr.label_resolver')) {
            /** @var \Drupal\address_tr\Service\AddressLabelResolver $resolver */
            $resolver = \Drupal::service('address_tr.label_resolver');
            $administrative_area_label = $resolver->getProvinceLabel($administrative_area);
            $locality_label = $resolver->getDistrictLabel($administrative_area, $locality);
            $dependent_locality_label = $resolver->getNeighbourhoodLabel($administrative_area, $locality, $dependent_locality);
        } else {
            $administrative_area_label = $administrative_area;
            $locality_label = $locality;
            $dependent_locality_label = $dependent_locality;
        }

        $location_parts = array_filter([
            $locality_label,
            $administrative_area_label,
            $sender_override['postal_code'] ?? '',
        ]);

        // Build full local address (Turkish style: Province, District, Neighborhood, Street)
        $local_parts = array_filter([
            $administrative_area_label,
            $locality_label,
            $dependent_locality_label,
            $sender_override['address_line1'] ?? '',
        ]);

        return [
            'name' => $sender_override['name'] ?? '',
            'address_line1' => $sender_override['address_line1'] ?? '',
            'address_line2' => $sender_override['address_line2'] ?? '',
            'locality' => $locality_label,
            'dependent_locality' => $dependent_locality_label,
            'administrative_area' => $administrative_area_label,
            'postal_code' => $sender_override['postal_code'] ?? '',
            'country_code' => $country_code,
            'country' => _commerce_shipping_label_get_country_name($country_code),
            'full_address' => implode('<br>', array_filter([
                implode(' ', $address_parts),
                implode(' ', $location_parts),
                _commerce_shipping_label_get_country_name($country_code),
            ])),
            'full_local_address' => implode(', ', $local_parts),
        ];
    }

    // Get from store.
    if ($order) {
        $store = $order->getStore();
        if ($store) {
            $address = $store->getAddress();
            if ($address) {
                $address_parts = array_filter([
                    $address->getAddressLine1(),
                    $address->getAddressLine2(),
                ]);

                // Get raw values.
                $administrative_area = $address->getAdministrativeArea();
                $locality = $address->getLocality();
                $dependent_locality = $address->getDependentLocality();
                $country_code = $address->getCountryCode();

                // For Turkish addresses, resolve codes to labels.
                if ($country_code === 'TR' && \Drupal::hasService('address_tr.label_resolver')) {
                    /** @var \Drupal\address_tr\Service\AddressLabelResolver $resolver */
                    $resolver = \Drupal::service('address_tr.label_resolver');
                    $administrative_area_label = $resolver->getProvinceLabel($administrative_area);
                    $locality_label = $resolver->getDistrictLabel($administrative_area, $locality);
                    $dependent_locality_label = $resolver->getNeighbourhoodLabel($administrative_area, $locality, $dependent_locality);
                } else {
                    $administrative_area_label = $administrative_area;
                    $locality_label = $locality;
                    $dependent_locality_label = $dependent_locality;
                }

                $location_parts = array_filter([
                    $locality_label,
                    $administrative_area_label,
                    $address->getPostalCode(),
                ]);

                // Build full local address (Turkish style)
                $local_parts = array_filter([
                    $administrative_area_label,
                    $locality_label,
                    $dependent_locality_label,
                    $address->getAddressLine1(),
                ]);

                return [
                    'name' => $store->getName(),
                    'address_line1' => $address->getAddressLine1(),
                    'address_line2' => $address->getAddressLine2(),
                    'locality' => $locality_label,
                    'dependent_locality' => $dependent_locality_label,
                    'administrative_area' => $administrative_area_label,
                    'postal_code' => $address->getPostalCode(),
                    'country_code' => $country_code,
                    'country' => _commerce_shipping_label_get_country_name($country_code),
                    'full_address' => implode('<br>', array_filter([
                        implode(' ', $address_parts),
                        implode(' ', $location_parts),
                        _commerce_shipping_label_get_country_name($country_code),
                    ])),
                    'full_local_address' => implode(', ', $local_parts),
                ];
            }
        }
    }

    return [];
}

/**
 * Gets recipient information from shipping profile.
 *
 * @param \Drupal\commerce_order\Entity\OrderInterface|null $order
 *   The order entity.
 * @param \Drupal\commerce_shipping\Entity\ShipmentInterface|null $shipment
 *   The shipment entity.
 *
 * @return array
 *   Array of recipient information.
 */
function _commerce_shipping_label_get_recipient_info($order, $shipment): array
{
    $profile = NULL;

    // Try to get shipping profile from shipment first.
    if ($shipment) {
        $profile = $shipment->getShippingProfile();
    }

    // Fallback to order's shipments.
    if (!$profile && $order && $order->hasField('shipments')) {
        $shipments = $order->get('shipments')->referencedEntities();
        if (!empty($shipments)) {
            $first_shipment = reset($shipments);
            $profile = $first_shipment->getShippingProfile();
        }
    }

    // Fallback to billing profile.
    if (!$profile && $order) {
        $profile = $order->getBillingProfile();
    }

    if (!$profile) {
        return [];
    }

    // Get address from profile.
    if (!$profile->hasField('address') || $profile->get('address')->isEmpty()) {
        return [];
    }

    $address = $profile->get('address')->first();
    if (!$address) {
        return [];
    }

    // Get phone if available.
    $phone = '';
    if ($profile->hasField('field_phone') && !$profile->get('field_phone')->isEmpty()) {
        $phone = $profile->get('field_phone')->value;
    } elseif ($profile->hasField('phone') && !$profile->get('phone')->isEmpty()) {
        $phone = $profile->get('phone')->value;
    }

    // Build recipient name.
    $name_parts = array_filter([
        $address->getGivenName(),
        $address->getFamilyName(),
    ]);
    $name = !empty($name_parts) ? implode(' ', $name_parts) : ($address->getOrganization() ?? '');

    $address_parts = array_filter([
        $address->getAddressLine1(),
        $address->getAddressLine2(),
    ]);

    // Get raw values.
    $administrative_area = $address->getAdministrativeArea();
    $locality = $address->getLocality();
    $dependent_locality = $address->getDependentLocality();
    $country_code = $address->getCountryCode();

    // For Turkish addresses, resolve codes to labels.
    if ($country_code === 'TR' && \Drupal::hasService('address_tr.label_resolver')) {
        /** @var \Drupal\address_tr\Service\AddressLabelResolver $resolver */
        $resolver = \Drupal::service('address_tr.label_resolver');
        $administrative_area_label = $resolver->getProvinceLabel($administrative_area);
        $locality_label = $resolver->getDistrictLabel($administrative_area, $locality);
        $dependent_locality_label = $resolver->getNeighbourhoodLabel($administrative_area, $locality, $dependent_locality);
    } else {
        $administrative_area_label = $administrative_area;
        $locality_label = $locality;
        $dependent_locality_label = $dependent_locality;
    }

    $location_parts = array_filter([
        $locality_label,
        $administrative_area_label,
        $address->getPostalCode(),
    ]);

    // Build full local address (Turkish style: Province, District, Neighborhood, Street)
    $local_parts = array_filter([
        $administrative_area_label,
        $locality_label,
        $dependent_locality_label,
        $address->getAddressLine1(),
    ]);

    return [
        'name' => $name,
        'organization' => $address->getOrganization(),
        'address_line1' => $address->getAddressLine1(),
        'address_line2' => $address->getAddressLine2(),
        'locality' => $locality_label,
        'dependent_locality' => $dependent_locality_label,
        'administrative_area' => $administrative_area_label,
        'postal_code' => $address->getPostalCode(),
        'country_code' => $country_code,
        'country' => _commerce_shipping_label_get_country_name($country_code),
        'phone' => $phone,
        'full_address' => implode('<br>', array_filter([
            implode(' ', $address_parts),
            implode(' ', $location_parts),
            _commerce_shipping_label_get_country_name($country_code),
        ])),
        'full_local_address' => implode(', ', $local_parts),
    ];
}

/**
 * Gets country name from country code.
 *
 * @param string $country_code
 *   The country code.
 *
 * @return string
 *   The country name.
 */
function _commerce_shipping_label_get_country_name(string $country_code): string
{
    if (empty($country_code)) {
        return '';
    }

    /** @var \Drupal\Core\Locale\CountryManagerInterface $country_manager */
    $country_manager = \Drupal::service('country_manager');
    $countries = $country_manager->getList();

    return $countries[$country_code] ?? $country_code;
}
