<?php

namespace Drupal\commerce_paratika_payment\PluginForm;

use Drupal\Core\Form\FormStateInterface;
use Drupal\commerce_payment\PluginForm\PaymentMethodAddForm as BasePaymentMethodAddForm;

/**
 * Payment method add form for the Paratika dummy gateway.
 *
 * This form accepts any credit card values without validation.
 */
class PaymentMethodAddForm extends BasePaymentMethodAddForm
{

    /**
     * {@inheritdoc}
     */
    protected function buildCreditCardForm(array $element, FormStateInterface $form_state)
    {
        $element = parent::buildCreditCardForm($element, $form_state);

        // Attach the CSS library.
        $element['#attached']['library'][] = 'commerce_paratika_payment/credit-card-form';

        // Add wrapper class for styling.
        $element['#attributes']['class'][] = 'credit-card-form';

        // Add a note that this is a test gateway.
        $element['test_note'] = [
            '#type' => 'item',
            '#markup' => '<div class="paratika-test-note"><span class="warning-icon">⚠️</span>' . $this->t('This is a test payment gateway. Any card details will be accepted.') . '</div>',
            '#weight' => -100,
        ];

        // Add card holder name field.
        $element['card_holder_name'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Card holder name'),
            '#attributes' => [
                'autocomplete' => 'cc-name',
                'placeholder' => $this->t('Name as it appears on card'),
            ],
            '#required' => TRUE,
            '#maxlength' => 255,
            '#weight' => -50,
        ];

        // Make card number full width.
        if (isset($element['number'])) {
            $element['number']['#attributes']['placeholder'] = $this->t('1234 5678 9012 3456');
        }

        // Wrap expiration and CVV in a container for same-row display.
        $element['expiration_cvv_wrapper'] = [
            '#type' => 'container',
            '#attributes' => [
                'class' => ['credit-card-form__expiration-cvv-wrapper'],
            ],
            '#weight' => 10,
        ];

        // Move expiration into wrapper with custom styling.
        if (isset($element['expiration'])) {
            $element['expiration']['#attributes']['class'][] = 'credit-card-form__expiration';
            $element['expiration']['#weight'] = 0;
            $element['expiration_cvv_wrapper']['expiration'] = $element['expiration'];
            unset($element['expiration']);
        }

        // Move CVV into wrapper.
        if (isset($element['security_code'])) {
            $element['security_code']['#weight'] = 1;
            $element['security_code']['#attributes']['placeholder'] = '123';
            $element['expiration_cvv_wrapper']['security_code'] = $element['security_code'];
            unset($element['security_code']);
        }

        return $element;
    }

    /**
     * {@inheritdoc}
     *
     * Skip all credit card validation - accept any values.
     */
    protected function validateCreditCardForm(array &$element, FormStateInterface $form_state)
    {
        // Get the submitted values.
        $values = $form_state->getValue($element['#parents']);

        // Set the card type based on the first digit or default to visa.
        $card_type = 'visa';
        if (!empty($values['number'])) {
            $first_digit = substr($values['number'], 0, 1);
            switch ($first_digit) {
                case '3':
                    $card_type = 'amex';
                    break;

                case '4':
                    $card_type = 'visa';
                    break;

                case '5':
                    $card_type = 'mastercard';
                    break;

                case '6':
                    $card_type = 'discover';
                    break;

                default:
                    $card_type = 'visa';
            }
        }

        // Set the card type without any validation.
        $form_state->setValueForElement($element['type'], $card_type);
    }

    /**
     * {@inheritdoc}
     *
     * Handle the moved expiration and security_code fields.
     */
    protected function submitCreditCardForm(array $element, FormStateInterface $form_state)
    {
        // Get the current values.
        $values = $form_state->getValue($element['#parents']);

        // Extract expiration and security_code from the wrapper if they're nested.
        if (isset($values['expiration_cvv_wrapper'])) {
            if (isset($values['expiration_cvv_wrapper']['expiration'])) {
                $values['expiration'] = $values['expiration_cvv_wrapper']['expiration'];
            }
            if (isset($values['expiration_cvv_wrapper']['security_code'])) {
                $values['security_code'] = $values['expiration_cvv_wrapper']['security_code'];
            }
            unset($values['expiration_cvv_wrapper']);

            // Set the modified values back.
            $form_state->setValue($element['#parents'], $values);
        }

        // Call parent submit.
        parent::submitCreditCardForm($element, $form_state);
    }
}
