<?php

namespace Drupal\commerce_paratika_payment\Plugin\Commerce\PaymentGateway;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\commerce_payment\Attribute\CommercePaymentGateway;
use Drupal\commerce_payment\CreditCard;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Entity\PaymentMethodInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OnsitePaymentGatewayBase;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsRefundsInterface;
use Drupal\commerce_paratika_payment\PluginForm\PaymentMethodAddForm;
use Drupal\commerce_price\Price;

/**
 * Provides the Paratika on-site payment gateway (dummy).
 *
 * This is a test/dummy gateway that accepts any credit card details
 * and immediately marks payments as completed.
 */
#[CommercePaymentGateway(
    id: "paratika_onsite",
    label: new TranslatableMarkup("Paratika (On-site)"),
    display_label: new TranslatableMarkup("Credit Card"),
    forms: [
        "add-payment-method" => PaymentMethodAddForm::class,
    ],
    payment_method_types: ["credit_card"],
    credit_card_types: [
        "amex",
        "dinersclub",
        "discover",
        "jcb",
        "maestro",
        "mastercard",
        "visa",
    ],
    requires_billing_information: FALSE,
)]
class ParatikaOnsite extends OnsitePaymentGatewayBase implements SupportsRefundsInterface
{

    /**
     * {@inheritdoc}
     */
    public function defaultConfiguration()
    {
        return [
            'store_payment_methods' => FALSE,
        ] + parent::defaultConfiguration();
    }

    /**
     * {@inheritdoc}
     */
    public function buildConfigurationForm(array $form, FormStateInterface $form_state)
    {
        $form = parent::buildConfigurationForm($form, $form_state);

        $form['store_payment_methods'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Store payment methods for reuse'),
            '#description' => $this->t('When enabled, customer credit card details (last 4 digits only) will be saved for future purchases.'),
            '#default_value' => $this->configuration['store_payment_methods'],
        ];

        // Add button to delete stored payment methods.
        $stored_count = $this->getStoredPaymentMethodsCount();
        $form['stored_payment_methods'] = [
            '#type' => 'details',
            '#title' => $this->t('Stored Payment Methods'),
            '#open' => $stored_count > 0,
        ];

        $form['stored_payment_methods']['count'] = [
            '#type' => 'item',
            '#markup' => $this->t('There are currently @count stored payment methods for this gateway.', [
                '@count' => $stored_count,
            ]),
        ];

        $form['stored_payment_methods']['delete_all'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Delete all stored payment methods'),
            '#description' => $this->t('Check this box and save to delete all @count stored payment methods. This action cannot be undone.', [
                '@count' => $stored_count,
            ]),
            '#default_value' => FALSE,
        ];

        return $form;
    }

    /**
     * {@inheritdoc}
     */
    public function submitConfigurationForm(array &$form, FormStateInterface $form_state)
    {
        parent::submitConfigurationForm($form, $form_state);

        $values = $form_state->getValue($form['#parents']);
        $this->configuration['store_payment_methods'] = !empty($values['store_payment_methods']);

        // Delete stored payment methods if requested.
        if (!empty($values['stored_payment_methods']['delete_all'])) {
            $deleted_count = $this->deleteAllStoredPaymentMethods();
            \Drupal::messenger()->addStatus($this->t('Deleted @count stored payment methods.', [
                '@count' => $deleted_count,
            ]));
        }
    }

    /**
     * Get the count of stored payment methods for this gateway.
     *
     * @return int
     *   The number of stored payment methods.
     */
    protected function getStoredPaymentMethodsCount(): int
    {
        $payment_gateway_id = $this->parentEntity ? $this->parentEntity->id() : NULL;
        if (!$payment_gateway_id) {
            return 0;
        }

        $storage = \Drupal::entityTypeManager()->getStorage('commerce_payment_method');
        $query = $storage->getQuery()
            ->condition('payment_gateway', $payment_gateway_id)
            ->accessCheck(FALSE);

        return (int) $query->count()->execute();
    }

    /**
     * Delete all stored payment methods for this gateway.
     *
     * @return int
     *   The number of deleted payment methods.
     */
    protected function deleteAllStoredPaymentMethods(): int
    {
        $payment_gateway_id = $this->parentEntity ? $this->parentEntity->id() : NULL;
        if (!$payment_gateway_id) {
            return 0;
        }

        $storage = \Drupal::entityTypeManager()->getStorage('commerce_payment_method');
        $payment_method_ids = $storage->getQuery()
            ->condition('payment_gateway', $payment_gateway_id)
            ->accessCheck(FALSE)
            ->execute();

        if (empty($payment_method_ids)) {
            return 0;
        }

        $payment_methods = $storage->loadMultiple($payment_method_ids);
        $count = count($payment_methods);
        $storage->delete($payment_methods);

        return $count;
    }

    /**
     * {@inheritdoc}
     */
    public function createPayment(PaymentInterface $payment, $capture = TRUE)
    {
        $this->assertPaymentState($payment, ['new']);
        $payment_method = $payment->getPaymentMethod();
        $this->assertPaymentMethod($payment_method);

        // This is a dummy gateway, so we just complete the payment immediately.
        // Generate a dummy remote ID.
        $remote_id = 'PARATIKA_' . time() . '_' . mt_rand(1000, 9999);

        $payment->setState('completed');
        $payment->setRemoteId($remote_id);
        $payment->save();
    }

    /**
     * {@inheritdoc}
     */
    public function createPaymentMethod(PaymentMethodInterface $payment_method, array $payment_details)
    {
        // Store the card details (only last 4 digits for number).
        $payment_method->card_type = $payment_details['type'] ?? 'visa';
        $payment_method->card_number = substr($payment_details['number'] ?? '0000', -4);
        $payment_method->card_exp_month = $payment_details['expiration']['month'] ?? date('m');
        $payment_method->card_exp_year = $payment_details['expiration']['year'] ?? date('Y');

        // Calculate expiration time.
        $expires = CreditCard::calculateExpirationTimestamp(
            $payment_method->card_exp_month->value,
            $payment_method->card_exp_year->value
        );

        // Generate a dummy remote ID.
        $remote_id = 'PARATIKA_METHOD_' . time() . '_' . mt_rand(1000, 9999);

        $payment_method->setRemoteId($remote_id);
        $payment_method->setExpiresTime($expires);

        // Only store for reuse if the setting is enabled.
        if (empty($this->configuration['store_payment_methods'])) {
            $payment_method->setReusable(FALSE);
        }

        $payment_method->save();
    }

    /**
     * {@inheritdoc}
     */
    public function deletePaymentMethod(PaymentMethodInterface $payment_method)
    {
        // Simply delete the local entity, no remote API to call.
        $payment_method->delete();
    }

    /**
     * {@inheritdoc}
     */
    public function refundPayment(PaymentInterface $payment, ?Price $amount = NULL)
    {
        $this->assertPaymentState($payment, ['completed', 'partially_refunded']);
        // If not specified, refund the entire amount.
        $amount = $amount ?: $payment->getAmount();
        $this->assertRefundAmount($payment, $amount);

        $old_refunded_amount = $payment->getRefundedAmount();
        $new_refunded_amount = $old_refunded_amount->add($amount);
        if ($new_refunded_amount->lessThan($payment->getAmount())) {
            $payment->setState('partially_refunded');
        } else {
            $payment->setState('refunded');
        }

        $payment->setRefundedAmount($new_refunded_amount);
        $payment->save();
    }
}
