<?php

namespace Drupal\commerce_paratika\PluginForm;

use Drupal\commerce_payment\PluginForm\PaymentGatewayFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides the payment form for Paratika.
 */
class ParatikaPaymentForm extends PaymentGatewayFormBase
{

    /**
     * The Paratika API client.
     *
     * @var \Drupal\commerce_paratika\ParatikaApiClient
     */
    protected $apiClient;

    /**
     * The private temp store.
     *
     * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
     */
    protected $tempStoreFactory;

    /**
     * The request stack.
     *
     * @var \Symfony\Component\HttpFoundation\RequestStack
     */
    protected $requestStack;

    /**
     * The logger.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container)
    {
        $instance = new static();
        $instance->apiClient = $container->get('commerce_paratika.api_client');
        $instance->tempStoreFactory = $container->get('tempstore.private');
        $instance->requestStack = $container->get('request_stack');
        $instance->logger = $container->get('logger.factory')->get('commerce_paratika');
        return $instance;
    }

    /**
     * Set the API client.
     */
    public function setApiClient($api_client)
    {
        $this->apiClient = $api_client;
    }

    /**
     * {@inheritdoc}
     */
    public function buildConfigurationForm(array $form, FormStateInterface $form_state)
    {
        // API client is already set via create() or setApiClient().

    // Get the order from the payment entity.
        /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
        $payment = $this->entity;
        /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
        $order = $payment->getOrder();

        /** @var \Drupal\commerce_paratika\Plugin\Commerce\PaymentGateway\Paratika $gateway */
        $gateway = $this->plugin;
        $credentials = $gateway->getCredentials();

        // Get configuration.
        $config = $gateway->getConfiguration();
        $enable_3d = isset($config['enable_3d']) ? $config['enable_3d'] : TRUE;
        $method_3ds = isset($config['3ds_method']) ? $config['3ds_method'] : 'popup';
        $enable_installments = isset($config['enable_installments']) ? $config['enable_installments'] : FALSE;

        // Check for temp store data from single page checkout.
        $temp_store = $this->tempStoreFactory->get('commerce_paratika');
        $temp_data = $temp_store->get('order_' . $order->id() . '_payment_data');

        // Attach the library for JavaScript and CSS.
        $form['#attached']['library'][] = 'commerce_paratika/paratika_payment';

        // Create session token with ALL required parameters from WooCommerce.
        $session_response = $this->createSessionToken($order, $credentials);

        if (empty($session_response) || $session_response['responseCode'] !== '00') {
            $error_msg = $session_response['errorMsg'] ?? 'Unknown error';
            $form['error'] = [
                '#markup' => '<div class="messages messages--error">' . $this->t('Unable to initialize payment: @error', ['@error' => $error_msg]) . '</div>',
            ];
            return $form;
        }

        $session_token = $session_response['sessionToken'];

        // Store session token in form state for later use.
        $form_state->set('paratika_session_token', $session_token);
        $form_state->set('paratika_order_id', $order->id());

        $form['#attributes']['class'][] = 'paratika-payment-form';
        $form['#attributes']['id'] = 'paratika-payment-form';

        $form['paratika_container'] = [
            '#type' => 'container',
            '#attributes' => ['class' => ['paratika-payment-container']],
        ];

        $form['paratika_container']['card_holder'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Cardholder Name'),
            '#required' => TRUE,
            '#default_value' => $temp_data['card_holder'] ?? '',
            '#attributes' => [
                'id' => 'cc-name',
                'autocomplete' => 'cc-name',
                'placeholder' => $this->t('Name on card'),
                'class' => ['paratika-card-holder'],
            ],
        ];

        $form['paratika_container']['card_number'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Card Number'),
            '#required' => TRUE,
            '#default_value' => $temp_data['card_number'] ?? '',
            '#attributes' => [
                'id' => 'cc-number',
                'autocomplete' => 'cc-number',
                'inputmode' => 'numeric',
                'pattern' => '[0-9]*',
                'placeholder' => '1234 5678 9012 3456',
                'maxlength' => 19,
                'class' => ['paratika-card-number'],
            ],
        ];

        // Container for Expiry and CVV to display in one row.
        $form['paratika_container']['expiry_cvv_wrapper'] = [
            '#type' => 'container',
            '#attributes' => ['class' => ['paratika-expiry-cvv-wrapper']],
        ];

        $form['paratika_container']['expiry_cvv_wrapper']['expiry_container'] = [
            '#type' => 'container',
            '#attributes' => ['class' => ['paratika-expiry-container']],
        ];

        $form['paratika_container']['expiry_cvv_wrapper']['expiry_container']['card_expiry_month'] = [
            '#type' => 'select',
            '#title' => $this->t('Expiry Month'),
            '#required' => TRUE,
            '#default_value' => $temp_data['card_expiry_month'] ?? '',
            '#options' => [
                '' => $this->t('Month'),
                '01' => '01',
                '02' => '02',
                '03' => '03',
                '04' => '04',
                '05' => '05',
                '06' => '06',
                '07' => '07',
                '08' => '08',
                '09' => '09',
                '10' => '10',
                '11' => '11',
                '12' => '12',
            ],
            '#attributes' => [
                'id' => 'cc-exp-month',
                'class' => ['paratika-expiry-month'],
                'autocomplete' => 'cc-exp-month',
            ],
        ];

        $current_year = date('Y');
        $years = [];
        for ($i = 0; $i < 20; $i++) {
            $year = $current_year + $i;
            $years[$year] = $year;
        }

        $form['paratika_container']['expiry_cvv_wrapper']['expiry_container']['card_expiry_year'] = [
            '#type' => 'select',
            '#title' => $this->t('Expiry Year'),
            '#required' => TRUE,
            '#default_value' => $temp_data['card_expiry_year'] ?? '',
            '#options' => ['' => $this->t('Year')] + $years,
            '#attributes' => [
                'id' => 'cc-exp-year',
                'class' => ['paratika-expiry-year'],
                'autocomplete' => 'cc-exp-year',
            ],
        ];

        $form['paratika_container']['expiry_cvv_wrapper']['card_cvv'] = [
            '#type' => 'textfield',
            '#title' => $this->t('CVV'),
            '#required' => TRUE,
            '#default_value' => $temp_data['card_cvv'] ?? '',
            '#attributes' => [
                'id' => 'cc-csc',
                'autocomplete' => 'cc-csc',
                'inputmode' => 'numeric',
                'pattern' => '[0-9]*',
                'placeholder' => '123',
                'maxlength' => 4,
                'class' => ['paratika-cvv'],
            ],
        ];

        // Installment selection.
        if ($enable_installments) {
            $form['paratika_container']['installments'] = [
                '#type' => 'select',
                '#title' => $this->t('Installments'),
                '#options' => [
                    '1' => $this->t('Single payment'),
                ],
                '#default_value' => $temp_data['installments'] ?? '1',
                '#attributes' => ['class' => ['paratika-installments']],
                '#prefix' => '<div id="paratika-installments-wrapper">',
                '#suffix' => '</div>',
            ];
        } else {
            $form['paratika_container']['installments'] = [
                '#type' => 'value',
                '#value' => '1',
            ];
        }

        // Hidden fields for Paratika.
        $form['paratika_container']['session_token'] = [
            '#type' => 'hidden',
            '#value' => $session_token,
            '#attributes' => ['class' => ['paratika-session-token']],
        ];

        // 3DS return URLs.
        $return_route_params = ['commerce_order' => $order->id()];
        $cancel_route_params = ['commerce_order' => $order->id()];

        if ($method_3ds === 'popup') {
            $return_route_params['is_popup'] = 1;
            $cancel_route_params['is_popup'] = 1;
        }

        $return_url = Url::fromRoute('commerce_paratika.3ds_return', $return_route_params, ['absolute' => TRUE])->toString();
        $cancel_url = Url::fromRoute('commerce_paratika.3ds_cancel', $cancel_route_params, ['absolute' => TRUE])->toString();

        // Construct 3DS sale URL.
        $test_mode = $gateway->getMode() == 'test';
        $api_url = $this->apiClient->getApiUrl($test_mode);
        $sale_url = $api_url . '/post/sale3d/' . $session_token;

        // Pass settings to JavaScript.
        $form['#attached']['drupalSettings']['commerceParatika'] = [
            'sessionToken' => $session_token,
            'returnUrl' => $return_url,
            'cancelUrl' => $cancel_url,
            'apiUrl' => $api_url,
            'saleUrl' => $sale_url,
            'installmentsUrl' => Url::fromRoute('commerce_paratika.query_installments')->toString(),
            'orderId' => $order->id(),
            'amount' => $order->getTotalPrice()->getNumber(),
            'currency' => $order->getTotalPrice()->getCurrencyCode(),
            'enable3ds' => $enable_3d,
            'method3ds' => $method_3ds,
            'enableInstallments' => $enable_installments,
            'testMode' => $test_mode,
            'autoSubmit' => !empty($temp_data),
        ];

        // Add submit button.
        $form['actions'] = [
            '#type' => 'actions',
            'submit' => [
                '#type' => 'submit',
                '#value' => $this->t('Pay and complete purchase'),
                '#button_type' => 'primary',
                '#attributes' => ['class' => ['button', 'button--primary']],
            ],
        ];

        return $form;
    }

    /**
     * {@inheritdoc}
     */
    public function validateConfigurationForm(array &$form, FormStateInterface $form_state)
    {
        // Validation is handled by JavaScript and Paratika API.
    }

    /**
     * {@inheritdoc}
     */
    public function submitConfigurationForm(array &$form, FormStateInterface $form_state)
    {
        // The actual payment submission is handled by JavaScript posting to Paratika.
        // This method is called after successful 3DS authentication.
    }

    /**
     * Create a payment session token with ALL parameters from WooCommerce.
     *
     * @param \Drupal\commerce_order\Entity\OrderInterface $order
     *   The order.
     * @param array $credentials
     *   The merchant credentials.
     *
     * @return array|null
     *   The API response.
     */
    protected function createSessionToken($order, array $credentials)
    {
        $amount = $order->getTotalPrice();

        // Get billing address.
        $billing_profile = $order->getBillingProfile();
        $address = $billing_profile ? $billing_profile->get('address')->first() : NULL;

        $return_url = Url::fromRoute('commerce_paratika.3ds_return', [
            'commerce_order' => $order->id(),
        ], ['absolute' => TRUE])->toString();

        $cancel_url = Url::fromRoute('commerce_paratika.3ds_cancel', [
            'commerce_order' => $order->id(),
        ], ['absolute' => TRUE])->toString();

        // Build customer name.
        $customer_name = '';
        if ($address) {
            $given_name = $address->getGivenName() ?? '';
            $family_name = $address->getFamilyName() ?? '';
            $customer_name = trim($given_name . ' ' . $family_name);
        }
        if (empty($customer_name)) {
            $customer_name = $order->getEmail() ?? 'Customer';
        }

        // Build order items array (matching WooCommerce format).
        $order_items = [];
        foreach ($order->getItems() as $order_item) {
            $item_object = new \stdClass();
            $item_object->productCode = $order_item->id();
            $item_object->name = $order_item->getTitle();
            $item_object->description = $order_item->getTitle();
            $item_object->quantity = (int) $order_item->getQuantity();
            $item_object->amount = (float) $order_item->getAdjustedUnitPrice()->getNumber();
            $order_items[] = $item_object;
        }

        // Add shipping if exists (check if field exists first).
        if ($order->hasField('shipments') && !$order->get('shipments')->isEmpty()) {
            $shipment_object = new \stdClass();
            $shipment_object->productCode = 'SHIPPING';
            $shipment_object->name = 'Shipping';
            $shipment_object->description = 'Shipping';
            $shipment_object->quantity = 1;
            $shipment_object->amount = (float) $order->getTotalPrice()->getNumber() - (float) $order->getSubtotalPrice()->getNumber();
            if ($shipment_object->amount > 0) {
                $order_items[] = $shipment_object;
            }
        }

        // Get address details.
        $address_line = '';
        $city = '';
        $country = '';
        $postal_code = '34000'; // Default
        $phone = '';

        if ($address) {
            $address_line = trim(($address->getAddressLine1() ?? '') . ' ' . ($address->getAddressLine2() ?? ''));
            $city = $address->getLocality() ?? '';
            $country = $address->getCountryCode() ?? '';
            $postal_code = $address->getPostalCode();
            if (empty($postal_code)) {
                $postal_code = '34000';
            }
        }

        // Get phone from billing profile or order.
        if ($billing_profile && $billing_profile->hasField('phone')) {
            $phone = $billing_profile->get('phone')->value ?? '';
        }
        if (empty($phone)) {
            $phone = '0000000000'; // Default
        }

        // Get customer email.
        $customer_email = $order->getEmail() ?? 'customer@example.com';

        // Get customer IP.
        $customer_ip = $order->getIpAddress() ?? $this->requestStack->getCurrentRequest()->getClientIp() ?? '127.0.0.1';

        // Format amount with 2 decimal places.
        $amount_formatted = number_format($amount->getNumber(), 2, '.', '');

        // Build parameters array matching WooCommerce exactly.
        $session_params = [
            'MERCHANTUSER' => $credentials['merchant_user'],
            'MERCHANTPASSWORD' => $credentials['merchant_password'],
            'MERCHANT' => $credentials['merchant_id'],
            'MERCHANTPAYMENTID' => $order->id() . '-' . time(),
            'AMOUNT' => $amount_formatted,
            'CURRENCY' => $amount->getCurrencyCode(),
            'RETURNURL' => $return_url,
            'CUSTOMER' => $customer_name,
            'CUSTOMERNAME' => $customer_name,
            'CUSTOMEREMAIL' => $customer_email,
            'CUSTOMERIP' => $customer_ip,
            'CUSTOMERUSERAGENT' => $this->requestStack->getCurrentRequest()->headers->get('User-Agent') ?? 'Drupal Commerce',
            'NAMEONCARD' => $customer_name,
            'CUSTOMERPHONE' => $phone,
            'ORDERITEMS' => urlencode(json_encode($order_items)),
            'BILLTOADDRESSLINE' => $address_line,
            'BILLTOCITY' => $city,
            'BILLTOCOUNTRY' => $country,
            'BILLTOPOSTALCODE' => $postal_code,
            'BILLTOPHONE' => $phone,
            'SHIPTOADDRESSLINE' => $address_line,
            'SHIPTOCITY' => $city,
            'SHIPTOCOUNTRY' => $country,
            'SHIPTOPOSTALCODE' => $postal_code,
            'SHIPTOPHONE' => $phone,
            'SESSIONTYPE' => 'PAYMENTSESSION',
            'test_mode' => $credentials['test_mode'],
        ];

        $this->logger->info('Session Token Params: <pre>@params</pre>', ['@params' => print_r($session_params, TRUE)]);

        return $this->apiClient->createSessionToken($session_params);
    }
}
