<?php

namespace Drupal\commerce_paratika\PluginForm;

use Drupal\commerce_payment\PluginForm\PaymentGatewayFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the checkout payment form for Paratika.
 */
class ParatikaCheckoutForm extends PaymentGatewayFormBase
{

    /**
     * The private temp store factory.
     *
     * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
     */
    protected $tempStoreFactory;

    /**
     * Constructs a new ParatikaCheckoutForm object.
     *
     * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
     *   The private temp store factory.
     */
    public function __construct(PrivateTempStoreFactory $temp_store_factory)
    {
        $this->tempStoreFactory = $temp_store_factory;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container)
    {
        return new static(
            $container->get('tempstore.private')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function buildConfigurationForm(array $form, FormStateInterface $form_state)
    {
        $gateway = $this->plugin;
        $config = $gateway->getConfiguration();
        $enable_installments = $config['enable_installments'] ?? FALSE;

        $form['#attributes']['class'][] = 'paratika-checkout-form';
        $form['#attributes']['id'] = 'paratika-payment-form';

        // Reuse the same fields as the offsite form, but without the hidden fields and auto-submit logic.
        // We just want to collect data here.

        $form['card_holder'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Cardholder Name'),
            '#required' => TRUE,
            '#attributes' => [
                'id' => 'cc-name',
                'autocomplete' => 'cc-name',
                'placeholder' => $this->t('Name on card'),
                'class' => ['paratika-card-holder'],
            ],
        ];

        $form['card_number'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Card Number'),
            '#required' => TRUE,
            '#attributes' => [
                'id' => 'cc-number',
                'autocomplete' => 'cc-number',
                'inputmode' => 'numeric',
                'pattern' => '[0-9]*',
                'placeholder' => '1234 5678 9012 3456',
                'maxlength' => 19,
                'class' => ['paratika-card-number'],
            ],
        ];

        $form['expiry_cvv_wrapper'] = [
            '#type' => 'container',
            '#attributes' => ['class' => ['paratika-expiry-cvv-wrapper']],
        ];

        $form['expiry_cvv_wrapper']['expiry_container'] = [
            '#type' => 'container',
            '#attributes' => ['class' => ['paratika-expiry-container']],
        ];

        $form['expiry_cvv_wrapper']['expiry_container']['card_expiry_month'] = [
            '#type' => 'select',
            '#title' => $this->t('Expiry Month'),
            '#required' => TRUE,
            '#options' => [
                '' => $this->t('Month'),
                '01' => '01',
                '02' => '02',
                '03' => '03',
                '04' => '04',
                '05' => '05',
                '06' => '06',
                '07' => '07',
                '08' => '08',
                '09' => '09',
                '10' => '10',
                '11' => '11',
                '12' => '12',
            ],
            '#attributes' => [
                'id' => 'cc-exp-month',
                'class' => ['paratika-expiry-month'],
                'autocomplete' => 'cc-exp-month',
            ],
        ];

        $current_year = date('Y');
        $years = [];
        for ($i = 0; $i < 20; $i++) {
            $year = $current_year + $i;
            $years[$year] = $year;
        }

        $form['expiry_cvv_wrapper']['expiry_container']['card_expiry_year'] = [
            '#type' => 'select',
            '#title' => $this->t('Expiry Year'),
            '#required' => TRUE,
            '#options' => ['' => $this->t('Year')] + $years,
            '#attributes' => [
                'id' => 'cc-exp-year',
                'class' => ['paratika-expiry-year'],
                'autocomplete' => 'cc-exp-year',
            ],
        ];

        $form['expiry_cvv_wrapper']['card_cvv'] = [
            '#type' => 'textfield',
            '#title' => $this->t('CVV'),
            '#required' => TRUE,
            '#attributes' => [
                'id' => 'cc-csc',
                'autocomplete' => 'cc-csc',
                'inputmode' => 'numeric',
                'pattern' => '[0-9]*',
                'placeholder' => '123',
                'maxlength' => 4,
                'class' => ['paratika-cvv'],
            ],
        ];

        if ($enable_installments) {
            $form['installments'] = [
                '#type' => 'select',
                '#title' => $this->t('Installments'),
                '#options' => [
                    '1' => $this->t('Single payment'),
                ],
                '#default_value' => '1',
                '#attributes' => ['class' => ['paratika-installments']],
                '#prefix' => '<div id="paratika-installments-wrapper">',
                '#suffix' => '</div>',
            ];
        } else {
            $form['installments'] = [
                '#type' => 'value',
                '#value' => '1',
            ];
        }

        // Attach the library for JS (installments query, formatting).
        // We can reuse the same library as it handles the UI logic.
        $form['#attached']['library'][] = 'commerce_paratika/paratika_payment';
    
    // Pass settings for JS (installments URL, etc).
    // We need to construct these settings similar to ParatikaPaymentForm.
        /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
        // Note: In add-payment-method context, we might not have a payment entity yet, 
        // but the form is usually built with one if it's the payment information pane?
        // Actually, usually $this->entity is the PaymentMethod entity being created.
        // But Paratika doesn't use stored payment methods. 
        // Let's see if we can get the order from the route match or context.

        $order = \Drupal::routeMatch()->getParameter('commerce_order');
        if ($order) {
            $test_mode = $gateway->getMode() == 'test';

            $form['#attached']['drupalSettings']['commerceParatika'] = [
                'installmentsUrl' => Url::fromRoute('commerce_paratika.query_installments')->toString(),
                'amount' => $order->getTotalPrice()->getNumber(),
                'currency' => $order->getTotalPrice()->getCurrencyCode(),
                'merchantId' => $config['merchant_id'],
                'enableInstallments' => $enable_installments,
                'testMode' => $test_mode,
                // We don't need sessionToken or saleUrl here yet.
            ];
        }

        return $form;
    }

    /**
     * {@inheritdoc}
     */
    public function validateConfigurationForm(array &$form, FormStateInterface $form_state)
    {
        // Basic validation.
        $values = $form_state->getValue($form['#parents']);

        if (empty($values['card_number'])) {
            $form_state->setError($form['card_number'], $this->t('Card number is required.'));
        }
        // Add more validation as needed (Luhn algorithm, etc).
    }

    /**
     * {@inheritdoc}
     */
    public function submitConfigurationForm(array &$form, FormStateInterface $form_state)
    {
        // Store the values in PrivateTempStore.
        $values = $form_state->getValue($form['#parents']);
        $order = \Drupal::routeMatch()->getParameter('commerce_order');

        if ($order) {
            $temp_store = $this->tempStoreFactory->get('commerce_paratika');
            $temp_store->set('order_' . $order->id() . '_payment_data', [
                'card_holder' => $values['card_holder'],
                'card_number' => $values['card_number'],
                'card_expiry_month' => $values['expiry_cvv_wrapper']['expiry_container']['card_expiry_month'],
                'card_expiry_year' => $values['expiry_cvv_wrapper']['expiry_container']['card_expiry_year'],
                'card_cvv' => $values['expiry_cvv_wrapper']['card_cvv'],
                'installments' => $values['installments'],
            ]);
        }
    }
}
