<?php

namespace Drupal\commerce_paratika\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\PaymentGatewayBase;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsRefundsInterface;
use Drupal\commerce_paratika\PluginForm\ParatikaPaymentForm;
use Drupal\commerce_paratika\PluginForm\ParatikaCheckoutForm;
use Drupal\commerce_price\Price;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\commerce_payment\Attribute\CommercePaymentGateway;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides the Paratika payment gateway.
 */
#[CommercePaymentGateway(
    id: "paratika",
    label: new TranslatableMarkup("Paratika"),
    display_label: new TranslatableMarkup("Credit Card (Paratika)"),
    forms: [
        "offsite-payment" => ParatikaPaymentForm::class,
        "add-payment-method" => ParatikaCheckoutForm::class,
    ],
    requires_billing_information: FALSE,
)]
class Paratika extends PaymentGatewayBase implements OffsitePaymentGatewayInterface, SupportsRefundsInterface
{

    /**
     * The Paratika API client.
     *
     * @var \Drupal\commerce_paratika\ParatikaApiClient
     */
    protected $apiClient;

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
        $instance->apiClient = $container->get('commerce_paratika.api_client');
        return $instance;
    }

    /**
     * {@inheritdoc}
     */
    public function defaultConfiguration()
    {
        return [
            'merchant_id' => '',
            'merchant_user' => '',
            'merchant_password' => '',
            'enable_3d' => TRUE,
            '3ds_method' => 'popup',
            'enable_installments' => FALSE,
        ] + parent::defaultConfiguration();
    }

    /**
     * {@inheritdoc}
     */
    public function buildConfigurationForm(array $form, FormStateInterface $form_state)
    {
        $form = parent::buildConfigurationForm($form, $form_state);

        $form['merchant_id'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Merchant ID'),
            '#description' => $this->t('Your Paratika merchant ID.'),
            '#default_value' => $this->configuration['merchant_id'],
            '#required' => TRUE,
        ];

        $form['merchant_user'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Merchant User'),
            '#description' => $this->t('Your Paratika merchant user.'),
            '#default_value' => $this->configuration['merchant_user'],
            '#required' => TRUE,
        ];

        $form['merchant_password'] = [
            '#type' => 'password',
            '#title' => $this->t('Merchant Password'),
            '#description' => $this->t('Your Paratika merchant password. Leave blank to keep existing password.'),
            '#default_value' => '',
        ];

        if (!empty($this->configuration['merchant_password'])) {
            $form['merchant_password']['#description'] = $this->t('Your Paratika merchant password is set. Leave blank to keep existing password.');
        }

        $form['enable_3d'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Enable 3D Secure'),
            '#description' => $this->t('If enabled, payments will use 3D Secure authentication.'),
            '#default_value' => $this->configuration['enable_3d'],
        ];

        $form['3ds_method'] = [
            '#type' => 'select',
            '#title' => $this->t('3D Secure Method'),
            '#description' => $this->t('Choose how the 3D Secure page should be opened.'),
            '#options' => [
                'redirect' => $this->t('Redirect (Same Window)'),
                'popup' => $this->t('Pop-up Window'),
            ],
            '#default_value' => $this->configuration['3ds_method'],
            '#states' => [
                'visible' => [
                    ':input[name="configuration[paratika][enable_3d]"]' => ['checked' => TRUE],
                ],
            ],
        ];

        $form['enable_installments'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Enable Installments'),
            '#description' => $this->t('If enabled, users can choose installment options.'),
            '#default_value' => $this->configuration['enable_installments'],
        ];

        return $form;
    }

    /**
     * {@inheritdoc}
     */
    public function submitConfigurationForm(array &$form, FormStateInterface $form_state)
    {
        parent::submitConfigurationForm($form, $form_state);

        if (!$form_state->getErrors()) {
            $values = $form_state->getValue($form['#parents']);
            $this->configuration['merchant_id'] = $values['merchant_id'];
            $this->configuration['merchant_user'] = $values['merchant_user'];
            $this->configuration['enable_3d'] = $values['enable_3d'];
            $this->configuration['3ds_method'] = $values['3ds_method'];
            $this->configuration['enable_installments'] = $values['enable_installments'];

            // Only update password if a new one was provided.
            if (!empty($values['merchant_password'])) {
                $this->configuration['merchant_password'] = $values['merchant_password'];
            }
        }
    }

    /**
     * {@inheritdoc}
     */
    public function onReturn(OrderInterface $order, Request $request)
    {
        // This is called when user returns from 3DS authentication.
        // The actual payment creation is handled by the controller.
    }

    /**
     * {@inheritdoc}
     */
    public function onCancel(OrderInterface $order, Request $request)
    {
        $this->messenger()->addWarning($this->t('Payment was cancelled.'));
    }

    /**
     * {@inheritdoc}
     */
    public function onNotify(Request $request)
    {
        // Paratika doesn't use server-to-server notifications.
        // All communication happens via return/cancel URLs.
    }

    /**
     * {@inheritdoc}
     */
    public function getNotifyUrl()
    {
        // Not used for Paratika.
        return NULL;
    }

    /**
     * {@inheritdoc}
     */
    public function createPayment(PaymentInterface $payment, $capture = TRUE)
    {
        $this->assertPaymentState($payment, ['new']);
        $order = $payment->getOrder();

        // Get the transaction ID from the order data.
        $order_data = $order->getData('paratika');
        if (empty($order_data['pgTranId'])) {
            throw new PaymentGatewayException('Missing Paratika transaction ID.');
        }

        // Query the transaction to verify it was successful.
        $response = $this->apiClient->queryTransaction([
            'MERCHANT' => $this->configuration['merchant_id'],
            'MERCHANTUSER' => $this->configuration['merchant_user'],
            'MERCHANTPASSWORD' => $this->configuration['merchant_password'],
            'PGTRANID' => $order_data['pgTranId'],
            'test_mode' => $this->getMode() == 'test',
        ]);

        if (empty($response) || $response['responseCode'] !== '00') {
            throw new PaymentGatewayException('Payment verification failed.');
        }

        $transaction = $response['transactionList'][0] ?? NULL;
        if (!$transaction || $transaction['transactionStatus'] !== 'AP') {
            throw new PaymentGatewayException('Payment was not approved.');
        }

        // Update payment with transaction details.
        $payment->setState('completed');
        $payment->setRemoteId($order_data['pgTranId']);
        $payment->setRemoteState($transaction['transactionStatus']);
        $payment->save();
    }

    /**
     * {@inheritdoc}
     */
    public function refundPayment(PaymentInterface $payment, ?Price $amount = NULL)
    {
        $this->assertPaymentState($payment, ['completed', 'partially_refunded']);
        $amount = $amount ?: $payment->getAmount();
        $this->assertRefundAmount($payment, $amount);

        $response = $this->apiClient->refund([
            'MERCHANT' => $this->configuration['merchant_id'],
            'MERCHANTUSER' => $this->configuration['merchant_user'],
            'MERCHANTPASSWORD' => $this->configuration['merchant_password'],
            'PGTRANID' => $payment->getRemoteId(),
            'AMOUNT' => $amount->getNumber(),
            'CURRENCY' => $amount->getCurrencyCode(),
            'test_mode' => $this->getMode() == 'test',
        ]);

        if (empty($response) || $response['responseCode'] !== '00') {
            $error_msg = $response['responseMsg'] ?? 'Unknown error';
            throw new PaymentGatewayException('Refund failed: ' . $error_msg);
        }

        $old_refunded_amount = $payment->getRefundedAmount();
        $new_refunded_amount = $old_refunded_amount->add($amount);

        if ($new_refunded_amount->lessThan($payment->getAmount())) {
            $payment->setState('partially_refunded');
        } else {
            $payment->setState('refunded');
        }

        $payment->setRefundedAmount($new_refunded_amount);
        $payment->save();
    }

    /**
     * {@inheritdoc}
     */
    public function voidPayment(PaymentInterface $payment)
    {
        $this->assertPaymentState($payment, ['authorization']);

        $response = $this->apiClient->cancel([
            'MERCHANT' => $this->configuration['merchant_id'],
            'MERCHANTUSER' => $this->configuration['merchant_user'],
            'MERCHANTPASSWORD' => $this->configuration['merchant_password'],
            'PGTRANID' => $payment->getRemoteId(),
            'test_mode' => $this->getMode() == 'test',
        ]);

        if (empty($response) || $response['responseCode'] !== '00') {
            $error_msg = $response['responseMsg'] ?? 'Unknown error';
            throw new PaymentGatewayException('Void failed: ' . $error_msg);
        }

        $payment->setState('authorization_voided');
        $payment->save();
    }

    /**
     * Get the merchant credentials.
     *
     * @return array
     *   Array with merchant_id, merchant_user, merchant_password, test_mode.
     */
    public function getCredentials()
    {
        return [
            'merchant_id' => $this->configuration['merchant_id'],
            'merchant_user' => $this->configuration['merchant_user'],
            'merchant_password' => $this->configuration['merchant_password'],
            'test_mode' => $this->getMode() == 'test',
        ];
    }
}
