<?php

namespace Drupal\commerce_paratika\Plugin\Commerce\CheckoutPane;

use Drupal\commerce_checkout\Plugin\Commerce\CheckoutPane\CheckoutPaneBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\commerce_paratika\PluginForm\ParatikaCheckoutForm;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the Paratika payment pane.
 *
 * @CommerceCheckoutPane(
 *   id = "paratika_payment",
 *   label = @Translation("Secure Payment"),
 *   default_step = "order_information",
 *   wrapper_element = "fieldset",
 * )
 */
class ParatikaPaymentPane extends CheckoutPaneBase
{

    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * The service container.
     *
     * @var \Symfony\Component\DependencyInjection\ContainerInterface
     */
    protected $container;

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition, $checkout_flow = NULL)
    {
        $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition, $checkout_flow);
        $instance->entityTypeManager = $container->get('entity_type.manager');
        $instance->container = $container;
        return $instance;
    }

    /**
     * {@inheritdoc}
     */
    public function buildPaneForm(array $pane_form, FormStateInterface $form_state, array &$complete_form)
    {
        // Check if Paratika is the selected payment gateway.
        // This is tricky because the selection happens in another pane (payment_information).
        // We can try to inspect the user input or the order's current payment gateway.

        // For now, we will render it but might need JS to toggle visibility based on payment method selection.
        // Or we assume Paratika is the only/main gateway for now as per user context.

        // Let's try to find the Paratika gateway.
        $storage = $this->entityTypeManager->getStorage('commerce_payment_gateway');
        $gateways = $storage->loadByProperties(['plugin' => 'paratika']);
        $gateway = reset($gateways);

        // Don't render the form if the gateway doesn't exist or is disabled.
        if (!$gateway || !$gateway->status()) {
            return $pane_form;
        }

        $plugin_form = ParatikaCheckoutForm::create($this->container);
        $plugin_form->setPlugin($gateway->getPlugin());

        $pane_form = $plugin_form->buildConfigurationForm($pane_form, $form_state);

        // Add a container class for easier JS targeting if needed.
        $pane_form['#attributes']['class'][] = 'paratika-payment-pane';

        return $pane_form;
    }

    /**
     * {@inheritdoc}
     */
    public function validatePaneForm(array &$pane_form, FormStateInterface $form_state, array &$complete_form)
    {
        if (!$this->isParatikaSelected($form_state)) {
            return;
        }

        $storage = $this->entityTypeManager->getStorage('commerce_payment_gateway');
        $gateways = $storage->loadByProperties(['plugin' => 'paratika']);
        $gateway = reset($gateways);

        if ($gateway) {
            $plugin_form = ParatikaCheckoutForm::create($this->container);
            $plugin_form->setPlugin($gateway->getPlugin());
            $plugin_form->validateConfigurationForm($pane_form, $form_state);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function submitPaneForm(array &$pane_form, FormStateInterface $form_state, array &$complete_form)
    {
        if (!$this->isParatikaSelected($form_state)) {
            return;
        }

        $storage = $this->entityTypeManager->getStorage('commerce_payment_gateway');
        $gateways = $storage->loadByProperties(['plugin' => 'paratika']);
        $gateway = reset($gateways);

        if ($gateway) {
            $plugin_form = ParatikaCheckoutForm::create($this->container);
            $plugin_form->setPlugin($gateway->getPlugin());
            $plugin_form->submitConfigurationForm($pane_form, $form_state);
        }
    }

    /**
     * Checks if Paratika is the selected payment method.
     */
    protected function isParatikaSelected(FormStateInterface $form_state)
    {
        $payment_values = $form_state->getValue('payment_information');
        $selected_method = $payment_values['payment_method'] ?? NULL;

        if (!$selected_method) {
            $input = $form_state->getUserInput();
            $selected_method = $input['payment_information']['payment_method'] ?? NULL;
        }

        // The value is the gateway ID. Our gateway ID is 'paratika'.
        return $selected_method === 'paratika';
    }
}
